#include "types-and-variables.h"
#include <stdlib.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>

OCXHANDLE			OCX_Handle = 0;
OCXHANDLE			OCX_Obj_Handle = 0;
OCXHANDLE			Conn_Handle = 0;
OCXCIPIDOBJ		ID_Obj;
volatile int	Backplane_Connected = FALSE;
volatile int	Backplane_Connecting = FALSE;
volatile int	PLC_mode = 0;
volatile int	PLC_DB_Initialized = FALSE;

OCXTAGDBHANDLE 	Tag_Db_Handle = 0;

// number of symbols in controller data base
WORD			Number_Of_Symbols = 0;

// controls exit from main program loop
volatile WORD	Done = FALSE;

volatile int exit_code = -1;
volatile const char* current_status = "Unknown";

static const unsigned char MVI_EDS_C8_FileData[] = {
#include "MVI56E_LDM_EDS_gz.h"
};

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// Display( )
//
//		Input:
//			display_string - null terminated string to display
//
//	  Description:
//			Provide new display_string, or reset display scroll to beginning by providing
//			a non NULL display_string argument. Providing a NULL display_string argument will
//			cause the display to be scrolled one char.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

void Display(char * display_string)
{
  int			result;
  static char	last_string[100];
  static int	index 				= -1;
  char		dsply_msg[5];
  int 		i;
  char		buf[5];
	char		error_string[80];
	memset(error_string, 0, sizeof(error_string));

  if (display_string != NULL)
  {
    // set new display string
    if(strlen(display_string) >= 4)
    {
      strncpy(last_string, display_string, 100);
      index = 0;
    }
    else
    {
      // if display string len is short, clear display
      index = -1;
    }
  }
  else
  {
    // scroll previous string by one char.
    index++;
    if (index >= strlen(last_string))
      index = 0;
  }

  // form 4 character string to be displayed.
  if(index < 0)
  {
    strcpy(dsply_msg, "    ");
  }
  else
  {
    if (strlen(&(last_string[index])) >= 4)
    {
      strncpy(dsply_msg, &(last_string[index]), 4);
    }
    else
    {
      strcpy(dsply_msg, &(last_string[index]));
      for (i = strlen(&(last_string[index])); i < 4; i++)
      {
        dsply_msg[i] = ' ';
      }
    }
  }

  dsply_msg[4] = 0;
  result = OCXcip_SetDisplay(OCX_Handle, dsply_msg);

  if (result != OCX_SUCCESS)
  {
		tw_ldm_log(TW_LDM_LOG_WARNING, "Set Display failed: result = %d, message length = %d, text = [%s]",
      result, strlen(dsply_msg), dsply_msg);
  }

  result = OCXcip_GetDisplay(OCX_Handle, buf);
  if (result != OCX_SUCCESS)
		tw_ldm_log(TW_LDM_LOG_ERROR, "Get Display failed: result = %d", result);
  else if (strcmp(dsply_msg, buf) != 0)
		tw_ldm_log(TW_LDM_LOG_ERROR, "Get Display returned [%s], should be [%s]", dsply_msg, buf);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		Backplane_ConnectProc()
//			Input:
//				objhandle	-  Handle ID of registered object handle ( from backplane open call )
//				psConn		-  pointer to an OCXCIPCONNSTRUCT, which contains information about the specific
//								connection.
//
//			Output:
//				returns OCX error code indicating success or failure type of operation
//
//			Description:  Called by the Backplane driver during the backplane connection process.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

OCXCALLBACK 	Backplane_ConnectProc(OCXHANDLE objhandle, OCXCIPCONNSTRUC *psConn)
{
  int		result = OCX_SUCCESS;

	// make sure this is for the registered handle.
  if(OCX_Obj_Handle != objhandle)
  {
    result = OCX_ERR_BADPARAM;
  }
  else
  {
    switch (psConn->reason)
    {
    case OCX_CIP_CONN_OPEN:
      // A new connection request is being made. Validate the
      // parameters and determine whether to allow the connection.
      // Return OCX_SUCCESS if the connection is to be established,
      // or one of the extended error codes if not. See the sample
      // code for more details.

      Conn_Handle = psConn->connHandle;
      break;

    case OCX_CIP_CONN_OPEN_COMPLETE:
      // The connection has been successfully opened. If necessary,
      // call OCXcip_WriteConnected to initialize transmit data.

      Backplane_Connected = TRUE; /* indicate connection is now open */
			Backplane_Connecting = FALSE;
      break;

    case OCX_CIP_CONN_NULLOPEN:
      // New configuration data is being passed to the open connection.
      // Process the data as necessary and return success.
      break;

    case OCX_CIP_CONN_CLOSE:
      // This connection has been closed  inform the application
			Backplane_Connecting = FALSE;
      break;

    default:
      result = OCX_CIP_FAILURE;
			Backplane_Connecting = FALSE;
      break;

    } /* end switch( psConn->reason ) */
  }
	
  return (result);
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		Backplane_ServiceProc()
//
//			Input:
//				objhandle	-  Handle ID of registered object handle ( from backplane open call )
//				psServ		-  pointer to an OCXCIPCONNSTRUCT, which contains information about the specific
//								connection.
//
//			Output:
//				returns OCX error code indicating success or failure type of operation
//
//
//			Description:  The backplane driver calls the Backplane_ServiceProc callback function when a Class 1
//						scheduled connection request is made for the registered object instance specified by objHandle.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

OCXCALLBACK     Backplane_ServiceProc(OCXHANDLE objhandle, OCXCIPSERVSTRUC *psServ)
{
  if (OCX_Obj_Handle != objhandle)
  {
    /*
    ** Somehow the object handles don't match. This
    ** should never happen. If it does return error.
    */
		tw_ldm_log(TW_LDM_LOG_ERROR, "Internal error in function Backplane_ServiceProc: object handle mismatch");
    return (OCX_CIP_FAILURE);
  }

  // Select which instance is being accessed.
  // The application defines how each instance is defined.

  switch(psServ->instance)
  {
  case 1: // Instance 1
  case 2 :  // Instance 2
  	// Check serviceCode and attribute; perform
  	// requested service if appropriate
  	switch(psServ->serviceCode)
    {
    case OCX_CIP_SC_GET_ATTR_SINGLE:
			tw_ldm_log(TW_LDM_LOG_DEBUG, "Service rcv: OCX_CIP_SC_GET_ATTR_SINGLE");
      break;
    case OCX_CIP_SC_SET_ATTR_SINGLE:
			tw_ldm_log(TW_LDM_LOG_DEBUG, "Service rcv: OCX_CIP_SC_SET_ATTR_SINGLE");
      break;
    case OCX_CIP_SC_GET_MEMBER:
			tw_ldm_log(TW_LDM_LOG_DEBUG, "Service rcv: OCX_CIP_SC_GET_MEMBER");
      break;
    case OCX_CIP_SC_SET_MEMBER:
			tw_ldm_log(TW_LDM_LOG_DEBUG, "Service rcv: OCX_CIP_SC_SET_MEMBER");
      break;
    case OCX_CIP_SC_GENERIC_RW:
			tw_ldm_log(TW_LDM_LOG_DEBUG, "Service rcv: OCX_CIP_SC_GENERIC_RW");
      break;
    }

    break;

  default:
		{ 
			tw_ldm_log(TW_LDM_LOG_ERROR, "Internal error in function Backplane_ServiceProc: invalid instance");
			return (OCX_CIP_BAD_INSTANCE); // Invalid instance
		}
  }

  return (OCX_SUCCESS);
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		print_rack_information( )
//			Input:
//					none
//
//			Description:
//					Prints information about the rack in human readable form.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

void print_rack_information(void)
{
  int				result;
  int				i;
  int				rack_size;
  DWORD			occupancy;
  char			error_string[80];
  OCXCIPEXDEVOBJ	extended_dev_obj;
	memset(error_string, 0, sizeof(error_string));

  result = OCXcip_GetActiveNodeTable(OCX_Handle, &rack_size, &occupancy);
  if (result == OCX_SUCCESS)
  {
    printf("\nRack Size is %d\n", rack_size);
    printf(" e is empty slot, M has a module\n");
    printf(" slot 0 to highest slot\n");

    for (i = 0; i < rack_size; i++)
      printf("%2d-", i);
    printf("\n");

    for (i = 0; i < rack_size; i++)
    {
      if (occupancy & (1 << i))
        //  		if ( occupancy & ( 2 >> i ) )
          			printf(" M ");
      else
        printf(" e ");
    }
    printf("\n");
  }
  else
  {
    OCXcip_ErrorString(result, error_string);
    printf("\nGet Active Node Table failed, result %d:%s\n", result, error_string);
  }

  //
  //  read the extended device object information
  //

  result = OCXcip_GetExDevObject(OCX_Handle, (BYTE*) gl_app_config.plc_path, &extended_dev_obj, 5000);

  if (result == OCX_SUCCESS)
  {
    printf("Device Name: %s \n", extended_dev_obj.Name);
    printf("Description: %s\n", extended_dev_obj.Description);
    printf("Number of ports: %d \n", extended_dev_obj.NumPorts);

    for (i = 0; i < extended_dev_obj.NumPorts; i++)
      printf("    Port[%d] is used for %d\n", extended_dev_obj.PortList[i].PortNum, extended_dev_obj.PortList[i].PortUse);

  }
  else
  {
    OCXcip_ErrorString(result, error_string);
    printf("\nGet extended device object failed, result %d:%s\n", result, error_string);
  }
};

int tw_ldm_is_connected_to_plc(void)
{
	return (Backplane_Connected && (PLC_mode == 1)) ? 1 : 0;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		check_controller_status()
//			Input:
//					none
//
//			Description: Checks status of the controller and sets global variable to current status.
//			Optionally, logs current status into log system (brief or detailed, depending on the argument verbose_log)
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
void check_controller_status(char do_log, char verbose_log)
{
  int					result;
	static int last_result = OCX_SUCCESS;
  WORD				status;
  WORD				mod_status;
  static WORD	last_status  = 0;
  char				error_string[80];
	const char* fault_status = "";
	const char* device_mode = "";
	const char* key_position = "";
#define STATUS_STRIGN_LEN 128
	static char status_string[STATUS_STRIGN_LEN + 1] = { 0 };

	memset(error_string, 0, sizeof(error_string));

  // read controller status
  result = OCXcip_GetDeviceIdStatus( OCX_Handle,
    (BYTE*) gl_app_config.plc_path,
    &status,
    5000);

  if (result == OCX_SUCCESS)
  {
    //  only print if the status has changed.  To avoid flooding of the logging system.
    if(status != last_status)
    {
      last_status = status;

			switch (status & OCX_ID_STATUS_FAULT_STATUS_MASK)
			{
			case OCX_ID_STATUS_RCV_MINOR_FAULT:
				fault_status = "Recoverable minor Fault";
				break;
			case OCX_ID_STATUS_URCV_MINOR_FAULT:
				fault_status = "Unrecoverable minor Fault";
				break;
			case OCX_ID_STATUS_RCV_MAJOR_FAULT:
				fault_status = "Recoverable Major Fault";
				break;
			case OCX_ID_STATUS_URCV_MAJOR_FAULT:
				fault_status = "Unrecoverable Major Fault";
				break;
			default:
				fault_status = "OK";
				break;
			}

			switch (status & OCX_ID_STATUS_DEVICE_STATUS_MASK)
			{
			case OCX_ID_STATUS_FLASHUPDATE:
				device_mode = "Flash update in progress";
				break;
			case OCX_ID_STATUS_FLASHBAD:
				device_mode = "Flash is bad";
				break;
			case OCX_ID_STATUS_FAULTED:
				device_mode = "Faulted";
				PLC_mode = 3;
				break;
			case OCX_ID_STATUS_RUN:
				device_mode = "Run";
				PLC_mode = 1;
				result = OCXcip_SetModuleStatusWord(OCX_Handle, OCX_ID_STATUS_RUN, OCX_ID_STATUS_DEVICE_STATUS_MASK);
				if (result != OCX_SUCCESS)
				{
					OCXcip_ErrorString(result, error_string);
					tw_ldm_log(TW_LDM_LOG_ERROR, "Set Device Status word failed, result [%d:%s]", result, error_string);
				}
				else
				{
					result = OCXcip_GetModuleStatusWord(OCX_Handle, &mod_status);
					if (result != OCX_SUCCESS)
					{
						OCXcip_ErrorString(result, error_string);
						tw_ldm_log(TW_LDM_LOG_ERROR, "Get Device Status word failed, result [%d:%s]", result, error_string);
					}
				}
				break;
			case OCX_ID_STATUS_PROGRAM:
				device_mode = "Program";
				PLC_mode = 2;

				result = OCXcip_SetModuleStatusWord(OCX_Handle, OCX_ID_STATUS_PROGRAM, OCX_ID_STATUS_DEVICE_STATUS_MASK);
				if (result != OCX_SUCCESS)
				{
					OCXcip_ErrorString(result, error_string);
					tw_ldm_log(TW_LDM_LOG_ERROR, "Set Device Status word failed, result [%d:%s]", result, error_string);
				}
				else
				{
					result = OCXcip_GetModuleStatusWord(OCX_Handle, &mod_status);
					if (result != OCX_SUCCESS)
					{
						OCXcip_ErrorString(result, error_string);
						tw_ldm_log(TW_LDM_LOG_ERROR, "Get Device Status word failed, result [%d:%s]", result, error_string);
					}
				}
				break;
			default:
				device_mode = "unknown";
				PLC_mode = 4;
			}

			switch (status & OCX_ID_STATUS_KEY_SWITCH_MASK)
			{
			case OCX_ID_STATUS_KEY_RUN:
				key_position = "RUN";
				break;
			case OCX_ID_STATUS_KEY_PROGRAM:
				key_position = "PROGRAM";
				break;
			case OCX_ID_STATUS_KEY_REMOTE:
				key_position = "REMOTE";
				break;
			default:
				key_position = "unknown";
				break;
			}
			// if (strlen(fault_status) + strlen(device_mode) + strlen(key_position)
				 
			snprintf(status_string, STATUS_STRIGN_LEN, "%s,%s,%s", fault_status, device_mode, key_position);
			current_status = status_string;

			if (do_log)
			{
				tw_ldm_log(TW_LDM_LOG_NOTICE, "PLC status = 0x%x [%s], mode = [%s], key position = [%s]", status, fault_status, device_mode, key_position);

				if (verbose_log)
				{
					print_rack_information();
					print_database_symbols();
				}
			} 
    }
  }
  else
  {
		if (result != last_result)
		{
			if (do_log)
			{
				OCXcip_ErrorString(result, error_string);
				tw_ldm_log(TW_LDM_LOG_ERROR, "Get Device ID Status failed, result %d:%s", result, error_string);
			}
			last_result = result;
		}
  }
}


/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		print_cip_data_type()
//			Input:
//				data_type - CIP of simple data type
//
//			Description:  prints the English readable description of the CIP data type supplied.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

void print_cip_data_type(DWORD data_type)
{

  switch (data_type)
  {
  case  OCX_CIP_BOOL:
    printf("Boolean");
    break;
  case  OCX_CIP_SINT:
    printf("Signed 8-bit integer");
    break;
  case  OCX_CIP_INT:
    printf("Signed 16-bit integer");
    break;
  case  OCX_CIP_DINT:
    printf("Signed 32-bit integer");
    break;
  case  OCX_CIP_LINT:
    printf("Signed 64-bit integer");
    break;
  case  OCX_CIP_USINT:
    printf("Unsigned 8-bit integer");
    break;
  case  OCX_CIP_UINT:
    printf("Unsigned 16-bit integer");
    break;
  case  OCX_CIP_UDINT:
    printf("Unsigned 32-bit integer");
    break;
  case  OCX_CIP_ULINT:
    printf("Unsigned 64-bit integer");
    break;
  case  OCX_CIP_REAL:
    printf("32-bit floating point");
    break;
  case  OCX_CIP_LREAL:
    printf("64-bit floating point");
    break;
  case  OCX_CIP_BYTE:
    printf("bit string, 8-bits");
    break;
  case  OCX_CIP_WORD:
    printf("bit string, 16-bits");
    break;
  case  OCX_CIP_DWORD:
    printf("bit string, 32-bits");
    break;
  case  OCX_CIP_LWORD:
    printf("bit string, 64-bits");
    break;
	case OCX_CIP_STRING82:
		printf("STRING type");
    break;
	case OCX_CIP_TAGDB_DATYPE_NORM_STRING:
		printf("NSTRING type");
		break;
  default:
    printf("unknown type 0x%x", (int) data_type);
    break;
  }

}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	print_structure_info()
//			Input:
//					pre_string	- A string to print preceding any output, allows easier reading of recursive
//									calls
//					struct_id	- The id of the structure data type in question.
//
//			Description:  Prints human readable information about the structure specified.  Since structure
//						definitions may be nested, this routine may call itself.  The pre_string is used to
//						indent the printing of nested structure information.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

void print_structure_info(char * pre_string, int struct_id)
{
  int						result;
  OCXCIPTAGDBSTRUCT		struct_info;
  OCXCIPTAGDBSTRUCTMBR	member_info;
  int						member_id;
  char					error_string[80];
  char					new_pre_string[40];

	memset(error_string, 0, sizeof(error_string));

  result = OCXcip_GetStructInfo(OCX_Handle, Tag_Db_Handle, struct_id, &struct_info);

  if (result == OCX_SUCCESS)
  {
    printf( "%s Structure: \"%s\": type num: %ld: number of members %d  size: %ld\n",
      pre_string,
      struct_info.name,
      struct_info.daType,
      struct_info.numMbr,
      struct_info.daSize);

    // display access information

    printf("%s Access-IO code %d\"%c\"\n", pre_string, struct_info.ioType, struct_info.ioType);


    // display information on the each of the members of the structure

      for(member_id = 0 ; member_id < struct_info.numMbr ; member_id++)
    {
      result = OCXcip_GetStructMbrInfo( OCX_Handle,
        Tag_Db_Handle,
        struct_id,
        member_id,
        &member_info);
      if (result != OCX_SUCCESS)
      {
        OCXcip_ErrorString(result, error_string);
        printf( "%s error reading MEMBER info for Member id = %d; error %d:%s\n",
          pre_string,
          member_id,
          result,
          error_string);
      }
      else
      {
        printf("%s M; %20s", pre_string, member_info.name);

        // if an array, show dimension
        if(member_info.arrDim)
        	printf("[%ld]", member_info.arrDim);
        else
        	printf("     "); 	// keep alignment

        printf(" : offset: %5ld; size %5ld; ", member_info.daOfs, member_info.eleSize);

        if (member_info.hStruct)
        {
          printf("Struct id %d::\n", member_info.hStruct);
          strcpy(new_pre_string, pre_string);
          strcat(new_pre_string, ":-");
          print_structure_info(new_pre_string, member_info.hStruct);
        }
        else
        {
          printf("Data Type: ");
          print_cip_data_type(member_info.daType);
          printf("\n");
        }

        if (member_info.fAttr & OCXCIPTAGDBSTRUCTMBR_ATTR_ALIAS)
        {
          printf("%s ALIAS for member %d\n", pre_string, member_info.baseMbrId);
        }

        // this information is not displayed
        // WORD    bitId;                  // Bit ID if daType is 0x00C1
        // BYTE    dispFmt;                // Recommended display format
      }
    }
  }
  else
  {
    OCXcip_ErrorString(result, error_string);
    printf( "%s error reading structure info for structure id = %d; error %d:%s\n",
      pre_string,
      struct_id,
      result,
      error_string);
  }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		print_database_symbols()
//			Input:
//					none
//
//			Description:	Prints the names of all the symbols in the Tag database.
//							The globals Tag_Db_Handle refers to an open tag dbase, and
//							Number_Of_Symbols gives the number of symbols in the dbase.
//							These are set by previous calls to open and build the dbase
//							from the tags in a controller.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

void print_database_symbols(void)
{
  int					result;
  WORD				symbol_id;
  OCXCIPTAGDBSYM		symbol_info;
  char				error_string[80];
	memset(error_string, 0, sizeof(error_string));

  printf("showing symbol information for %d symbols\n", Number_Of_Symbols);

  // for every Symbol in the data base, read the symbol information and print it.
  for(symbol_id = 0 ; symbol_id < Number_Of_Symbols ; symbol_id++)
  {
    result = OCXcip_GetSymbolInfo(OCX_Handle, Tag_Db_Handle, symbol_id, &symbol_info);

    if (result == OCX_SUCCESS)
    {
      printf("\nSymbol %d name = \"%s\"", symbol_id, symbol_info.name);
      if (symbol_info.xDim)
      {
        printf("[%ld", symbol_info.xDim);

        if (symbol_info.yDim)
          printf(",%ld", symbol_info.yDim);
        if (symbol_info.zDim)
          printf(",%ld", symbol_info.zDim);
        printf("]");
      }

      // print size of element
      printf(" element Size = %ld\n", symbol_info.eleSize);

      if (!symbol_info.hStruct)
      {
        // non structure, so print simple data type

        printf("Simple type = ");
        print_cip_data_type(symbol_info.daType);
      }
      else
      {
        // this is a structure type, so print structure formation
        print_structure_info("", symbol_info.hStruct);
      }

      printf("\n");
    }
    else
    {
      OCXcip_ErrorString(result, error_string);
      printf( "error reading symbol info for symbol %d; error %d:%s\n",
        symbol_id,
        result,
        error_string);
    }
  }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		open_tag_dbase()
//			Input:
//					path_to_controller - null terminated string specifying the path to a controller.
//
//			Description:    Opens a Tag Database with the controller, checks the version, and rebuilds the
//						the local database if needed.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

int open_tag_dbase(char * path_to_controller)
{
  int					result;
  char				error_string[80];
	memset(error_string, 0, sizeof(error_string));

  // create tag data base
  result = OCXcip_CreateTagDbHandle( OCX_Handle,
    path_to_controller,
    DEFAULT_OCX_TIMEOUT,
    &Tag_Db_Handle);
  if (result == OCX_SUCCESS)
  {
    // set the Tag Data base options ( alignment, etc )
    //result = OCXcip_SetTagDbOptions(OCX_Handle, Tag_Db_Handle, OCX_CIP_TAGDBOPT_NORM_STRINGS, 4);
    //if (result != OCX_SUCCESS)
    //{
    //  OCXcip_ErrorString(result, error_string);
    //  printf("Set Tag Dbase options failed; returned %d:%s\n", result, error_string);
    //}

    // Make sure the Tag Data base match controller
    result = OCXcip_TestTagDbVer(OCX_Handle, Tag_Db_Handle);
    if ((result == OCX_ERR_OBJEMPTY) || (result == OCX_ERR_VERMISMATCH))
    {
      // rebuild data base by reading controller
      result = OCXcip_BuildTagDb(OCX_Handle, Tag_Db_Handle, &Number_Of_Symbols);
      if (result != OCX_SUCCESS)
      {
        OCXcip_ErrorString(result, error_string);
				tw_ldm_log(TW_LDM_LOG_ERROR, "Build of Tag Dbase failed. returned %d:%s", result, error_string);
      }
    }
		else if (result != OCX_SUCCESS)
		{
			OCXcip_ErrorString(result, error_string);
			tw_ldm_log(TW_LDM_LOG_ERROR, "TestTag Dbase Version returned %d:%s", result, error_string);
		}
  }
	else
	{
		OCXcip_ErrorString(result, error_string);
		tw_ldm_log(TW_LDM_LOG_ERROR, "OCXcip_CreateTagDbHandle function failed with error %d [%s]. Path to controller = [%s]", 
			result, error_string, path_to_controller);
	}
  if (result == OCX_SUCCESS)
    PLC_DB_Initialized = TRUE;
  else
    PLC_DB_Initialized = FALSE;
  
  return result;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	close_tag_dbase()
//			Input:
//					none
//			Output:
//					none
//
//			Description:  Closes the tag data base.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

void close_tag_dbase(void)
{
  int					result;
  char				error_string[80];
	memset(error_string, 0, sizeof(error_string));

  result = OCXcip_DeleteTagDbHandle( OCX_Handle,
    Tag_Db_Handle);

  if (result != OCX_SUCCESS)
  {
    OCXcip_ErrorString(result, error_string);
    tw_ldm_log(TW_LDM_LOG_ERROR, "Delete of Tag Dbase handle failed. returned [%d:%s]", result, error_string);
  }
  else
  {
    Tag_Db_Handle = (OCXTAGDBHANDLE) 0;
  }
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//	get_tag_info()
//			Input:
//					tag_name - the name of the controller tag for which information is desired.
//			Output:
//					tag_definition - tag access definition containing detailed info from controller
//									about this tag.
//
//			Description:  Gets information about the controller tag.
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

int  get_tag_info(char * tag_name, OCXCIPTAGACCESS * tag_definition)
{
  int					  result;
  OCXCIPTAGINFO tag_info;
  char				  error_string[80];
	memset(error_string, 0, sizeof(error_string));

  result = OCXcip_GetTagDbTagInfo( OCX_Handle, Tag_Db_Handle, tag_name, &tag_info);

  if (result == OCX_SUCCESS)
  {
    tag_definition->daType = tag_info.daType;
		if (tag_info.daType > 0xFFFF)
		{
			tw_ldm_log(TW_LDM_LOG_WARNING, "Tag [%s]: data type [0x%x] converted to [0x%x]", tag_name, tag_info.daType, tag_definition->daType);
		}
    tag_definition->eleSize = tag_info.eleSize;
  }
  else
  {
    OCXcip_ErrorString(result, error_string);
    tw_ldm_log(TW_LDM_LOG_ERROR, "OCXcip_GetTagDbTagInfo failed reading tag %s; error [%d: %s]", tag_name, result, error_string);
  }
	return result;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		open_backplane()
//			Input:
//					none
//
//			Output:
//					The global OCX_Handle is set if successful.
//
//			Description:
//					Opens a connection to the backplane driver, allow access to some of the module hardware
// 				including backplane communication
//
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
int open_backplane(void)
{
  int								result;
  OCXEDSFILEDATA  	eds_data;
	char							error_string[80];
	memset(error_string, 0, sizeof(error_string));

  result = OCXcip_Open(&OCX_Handle);

  if (result != OCX_SUCCESS)
  {
		OCXcip_ErrorString(result, error_string);
    tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane: call to OCXcip_Open returned 0x%x [%d, %s]", result, result, error_string);
    return result;
  }

  result = OCXcip_GetIdObject(OCX_Handle, &ID_Obj);
	if (result != OCX_SUCCESS)
	{
		OCXcip_ErrorString(result, error_string);
		tw_ldm_log(TW_LDM_LOG_ERROR, "OCXcip_GetIdObject failed: result = [%d, %s]", result, error_string);
	}
	else
	{
		tw_ldm_log(TW_LDM_LOG_DEBUG, "OCXcip_GetIdObject, serial # %lX", ID_Obj.SerialNo);
		tw_ldm_log(TW_LDM_LOG_DEBUG, "OCXcip_GetIdObject, in slot %d", ID_Obj.Slot);

		// set the module object identification
		ID_Obj.VendorID				= 309;  // Prosoft
		ID_Obj.DeviceType			= 12;   // communications adapter type
		ID_Obj.ProductCode		= 5020; // LDM module
		ID_Obj.MajorRevision	= 1; 		// major revision 1
		ID_Obj.MinorRevision	= 1; 		// minor revision 1

		strcpy((char *)ID_Obj.Name, "MVI56E-LDM");

		result = OCXcip_SetIdObject(OCX_Handle, (OCXCIPIDOBJ *)&ID_Obj);
		if (result != OCX_SUCCESS)
		{
			OCXcip_ErrorString(result, error_string);
			tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane: Set ID Object failed, error = [%d, %s]", result, error_string);
			return result;
		}

		// set embedded EDS file.
		eds_data.iInstanceId = 0xc8;
		eds_data.bMajorRevision = 1;
		eds_data.bMinorRevision = 1;
		eds_data.bFileEncoding = 1;  // Compressed
		eds_data.lFileSize = sizeof(MVI_EDS_C8_FileData);
		eds_data.pFileData = (BYTE *)MVI_EDS_C8_FileData;

		result = OCXcip_SetEmbeddedEDSFile(OCX_Handle, &eds_data);
		if (result != OCX_SUCCESS)
		{
			OCXcip_ErrorString(result, error_string);
			tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane: Failed to set the Embedded EDS file: [0x%X, %s]", result, error_string);
			return result;
		}

		// register call back functions.
		result = OCXcip_RegisterAssemblyObj(OCX_Handle, &OCX_Obj_Handle, 0L, Backplane_ConnectProc,	Backplane_ServiceProc);
		if (result != OCX_SUCCESS)
		{
			OCXcip_ErrorString(result, error_string);
			tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane: Assembly Registration failed, result = [%d, %s]", result, error_string);
			return result;
		}

		result = OCXcip_SetModuleStatusWord(OCX_Handle, OCX_MOD_STATUS_UNOWNED, OCX_MOD_STATUS_OWNED_MASK);
		if (result != OCX_SUCCESS)
		{
			OCXcip_ErrorString(result, error_string);
			tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane: Set Module Status, step 1 failed , result = [%d, %s]", result, error_string);
			return result;
		}
		OCXcip_SetModuleStatusWord(OCX_Handle, OCX_MOD_STATUS_CONFIGURED, OCX_MOD_STATUS_CONFIGURED_MASK);
		if (result != OCX_SUCCESS)
		{
			OCXcip_ErrorString(result, error_string);
			tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane: Set Module Status, step 2 failed , result = [%d, %s]", result, error_string);
			return result;
		}
		OCXcip_SetModuleStatusWord(OCX_Handle, OCX_MOD_STATUS_AWAIT_CONN, OCX_MOD_STATUS_STATE_MASK);
		if (result != OCX_SUCCESS)
		{
			OCXcip_ErrorString(result, error_string);
			tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane: Set Module Status, step 3 failed , result = [%d, %s]", result, error_string);
			return result;
		}
		OCXcip_SetModuleStatusWord(OCX_Handle, 0, OCX_MOD_STATUS_FAULT_MASK);
		if (result != OCX_SUCCESS)
		{
			OCXcip_ErrorString(result, error_string);
			tw_ldm_log(TW_LDM_LOG_ERROR, "open_backplane: Set Module Status, step 4 failed , result = [%d, %s]", result, error_string);
			return result;
		}
	}
  return result;
}

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
//		close_backplane()
//			Input:
//					none
//
//			Description:  Unregisters Assembly object, Closes tag database, and backplane object.
//
//
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
void close_backplane(void)
{
	int		result;
	char	error_string[80];
	memset(error_string, 0, sizeof(error_string));

  close_tag_dbase();
  // unregister assembly
	result = OCXcip_UnregisterAssemblyObj(OCX_Handle, OCX_Obj_Handle);
	if (result != OCX_SUCCESS)
	{
		OCXcip_ErrorString(result, error_string);
		tw_ldm_log(TW_LDM_LOG_ERROR, "close_backplane: Failed to unregister assembly, result = [%d, %s]", result, error_string);
	}
  // close cip handle
  result = OCXcip_Close(OCX_Handle);
	if (result != OCX_SUCCESS)
	{
		OCXcip_ErrorString(result, error_string);
		tw_ldm_log(TW_LDM_LOG_ERROR, "close_backplane: Failed to close handle, result = %d", result, error_string);
	}
	Backplane_Connected = FALSE;
}


int tw_ldm_get_status(char is_verbose, char** buffer, uint16_t max_size)
{
	const char* log_file_name = "/www/html/log/messages.txt";

	if (!is_verbose)
	{
		const char* status_to_return = (const char*) current_status;

		uint16_t len = (uint16_t)strlen(status_to_return);
		if (!*buffer)
		{
			*buffer = (char*)malloc(len + 1);
			if (*buffer)
				return -1;
		}
		if (len > max_size)
			len = max_size;
		(*buffer)[len] = 0;
		if (len > 0)
			strncpy(*buffer, status_to_return, len);
		return 0;
	}

	// If verbose, then return content of the log file.
	else
	{
		FILE* f = fopen(log_file_name, "r");
		long file_size;
		size_t read_bytes;

		uint16_t bytes_to_read;
		if (!f)
			return -1;
		fseek(f, 0, SEEK_END);
		file_size = ftell(f);

		if (max_size > 0 && file_size > max_size)
			bytes_to_read = max_size;
		else
		{
			if (file_size > 0x7FFE)
				bytes_to_read = 0x7FFE;
			else
				bytes_to_read = (uint16_t)file_size;
		}

		if (!*buffer)
		{
			*buffer = (char*)malloc(bytes_to_read + 1);
			if (!*buffer)
				return -1;
		}
		fseek(f, file_size - bytes_to_read, SEEK_SET);
		read_bytes = fread(*buffer, 1, bytes_to_read, f);
		fclose(f);

		(*buffer)[bytes_to_read] = 0;
		if (read_bytes == bytes_to_read)
			return 0;
		else
			return -1;
	}
}