/*
|============================================================================
|
|       Copyright (C) 2012 ProSoft Technology. All rights reserved.
|
|  File:             cfgparse.c
|
|  Class(es):        
|
|  Inherits From:    
|
|  Summary:          
|
|  Project(s):       PLX Utility
|
|  Subsystem:        Common
|
|  Contributors:     Henry Yu(HYU)
|
|  Description:      Config file parser
|
|  Notes:            
|
|
|============================================================================
|  Version     Date     Author  Change    Description
|----------------------------------------------------------------------------
|  Build     08/26/2012 HYU              Created.
|============================================================================
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>
#include "cfgparse.h"



#define MAX_LINE_LEN   500


#ifndef stricmp
  #define stricmp   strcasecmp
#endif

#ifndef strnicmp
  #define strnicmp(s1, s2, maxlen) strncasecmp(s1,s2,maxlen)
#endif


static int trim(char *s);

static ConfigSection * NewConfigSection(ConfigParser * me);
static void ConfigSection_Cleanup(ConfigSection * me);
static ConfigItem * AddNewConfigItem(ConfigSection * pSection, char * line);
static ConfigItem * ConfigParser_SeekItem(ConfigParser * me, const char * itemName, int * valPos);


//delimiters: 0 - any space char, '=', ':'
void ConfigParser_Init(ConfigParser * me, char delimiter)
{
    me->fptr = NULL;
    me->delimiter = delimiter;
    me->currentSection = NULL;
    LL_Init( &me->sectionList );
}

void ConfigParser_Cleanup(ConfigParser * me)
{
    ConfigSection * pSection = NULL;
    while( !LL_Empty( &me->sectionList ) )
    {
        pSection = (ConfigSection *)LL_FirstEntry (&me->sectionList);
        LL_RemoveDoubleLinkList(pSection);
        ConfigSection_Cleanup(pSection);
        free(pSection);
    }

    if (me->fptr != NULL)
    {
        fclose(me->fptr);
        me->fptr = NULL;
    }
}

int ConfigParser_LoadSection(ConfigParser * me, const char * fname, const char * sectionName)
{
    FILE * fptr;
    char buff[MAX_LINE_LEN+1];
    int foundSection = 0;
    char *cptr;
    ConfigSection * currentSection = NULL;

    if( (fptr=fopen(fname, "rt")) == NULL)
    {
        return 0;
    }

    me->currentSection = NULL;

    fseek(fptr, 0, SEEK_SET);	//start at beginning of file
    
    //try to reload all the items under the section
    while (fgets(buff, MAX_LINE_LEN, fptr) != NULL)
    {
        if (buff[0] == '\0' || buff[0] == '#' || buff[0] == '\n' || buff[0] == '\r')
            continue;
        
        /* kill the linefeed and any trailing whitespace */
        trim(buff);
        if (buff[0] == '\0')
            continue;
        
        if (!foundSection)
        {
            if (buff[0] == '[' && stricmp(buff, sectionName) == 0)
            {
                foundSection = 1;
                
                currentSection = NewConfigSection(me);
                if (currentSection != NULL)
                {
                    me->currentSection = currentSection;
                    
                    strncpy(currentSection->items.line, buff, MAX_CFG_LINE_LEN-1);
                    currentSection->items.line[MAX_CFG_LINE_LEN-1] = '\0';
                }
            }
            
            continue;
        }
        
        if (buff[0] == '[')
        {
            break;
        }
        
        cptr = strchr(buff, '#');  //find comment delimiter
        if(cptr != NULL)
        {
            *cptr = '\0'; //remove comment
        }
        
        AddNewConfigItem(currentSection, buff);
    }

    fclose(fptr);

    return 1;
}

static ConfigSection * NewConfigSection(ConfigParser * me)
{
    ConfigSection * pSection = (ConfigSection *)malloc(sizeof(ConfigSection));
    if (pSection != NULL)
    {
        LL_Init( &pSection->items );
        pSection->items.line[0] = '\0';
        LL_PutDoubleLinkList(&me->sectionList, pSection);
    }

    return pSection;
}

static void ConfigSection_Cleanup(ConfigSection * me)
{
    ConfigItem * pItem;
    while( !LL_Empty( &me->items ) )
    {
        pItem = (ConfigItem *)LL_FirstEntry (&me->items);
        LL_RemoveDoubleLinkList(pItem);
        free(pItem);
    }
}

static ConfigItem * AddNewConfigItem(ConfigSection * pSection, char * line)
{
    ConfigItem * pItem = (ConfigItem *)malloc(sizeof(ConfigItem));
    if (pItem != NULL)
    {
        strncpy(pItem->line, line, MAX_CFG_LINE_LEN-1);
        pItem->line[MAX_CFG_LINE_LEN-1] = '\0';
        
        LL_PutDoubleLinkList(&pSection->items, pItem);
    }

    return pItem;
}

int ConfigParser_Load(ConfigParser * me, const char * fname)
{
    char buff[MAX_LINE_LEN+1];
    char *cptr;
    FILE * fptr;
    ConfigSection * currentSection = NULL;

    if( (fptr=fopen(fname, "rt")) == NULL)
    {
        return 0;
    }

    fseek(fptr, 0, SEEK_SET);	//start at beginning of file

    while (fgets(buff, MAX_LINE_LEN, fptr) != NULL)
    {
        if (buff[0] == '\0' || buff[0] == '#' || buff[0] == '\n' || buff[0] == '\r')
            continue;
        
        /* kill the linefeed and any trailing whitespace */
        trim(buff);
        if (buff[0] == '\0')
        {
            continue;
        }

        if (buff[0] == '[')
        {
            currentSection = NewConfigSection(me);
            if (currentSection != NULL)
            {
#ifdef DEBUG
                printf("%s: Section Name: '%s'\n", __func__, buff);
#endif
                strncpy(currentSection->items.line, buff, MAX_CFG_LINE_LEN-1);
                currentSection->items.line[MAX_CFG_LINE_LEN-1] = '\0';
            }

            continue;
        }

        cptr = strchr(buff, '#');  //find comment delimiter
        if(cptr != NULL)
        {
            *cptr = '\0'; //remove comment
        }

        if (LL_Empty(&me->sectionList))
        {
            currentSection = NewConfigSection(me); //create a root section
        }

#ifdef DEBUG
        printf("%s: Section Item: '%s'\n", __func__, buff);
#endif

        AddNewConfigItem(currentSection, buff);
    }
    
    fclose(fptr);

    me->currentSection = NULL;
    if (!LL_Empty( &me->sectionList ))
    {
        me->currentSection = (ConfigSection *)LL_FirstEntry (&me->sectionList);
    }

    return 1;
}

int ConfigParser_Save(ConfigParser * me, const char * fname)
{
    FILE * fptr;

    if ((fptr = fopen(fname, "w")) == NULL)
    {
        return 0;
    }

    int ret = ConfigParser_Save2(me, fptr);

    fclose(fptr);

    sync();

    return ret;
}

int ConfigParser_Save2(ConfigParser * me, FILE *fptr)
{
    int fd;
    ConfigItem * pItem;

    ConfigSection * pSection = (ConfigSection *)LL_FirstEntry (&me->sectionList);
    while (!LL_EndOfList(&me->sectionList, pSection) )
    {
        if (pSection->items.line[0] != '\0')
        {
            fputs(pSection->items.line, fptr);
            fputs("\n", fptr);
        }

        pItem = (ConfigItem *)LL_FirstEntry (&pSection->items);
        while (!LL_EndOfList(&pSection->items, pItem) )
        {
            fputs(pItem->line, fptr);
            fputs("\n", fptr);

            pItem = LL_NextEntry(pItem);
        }
        
        pSection = LL_NextEntry(pSection);
    }

    fd = fileno(fptr);
    fchmod (fd, 0644);
    fchmod (fd, 0666);
    fsync(fd);

    return 1;
}


int ConfigParser_SeekSection(ConfigParser * me, const char * sectionName)
{
    me->currentSection = NULL;

    if ((sectionName == NULL || *sectionName == '\0') && !LL_Empty( &me->sectionList ))
    {
        me->currentSection = (ConfigSection *)LL_FirstEntry (&me->sectionList);
    }

    ConfigSection * pSection = (ConfigSection *)LL_FirstEntry (&me->sectionList);
    while (!LL_EndOfList(&me->sectionList, pSection) )
    {
        if (stricmp(pSection->items.line, sectionName) == 0)
        {
            me->currentSection = pSection;
            return 1;
        }

        pSection = LL_NextEntry(pSection);
    }

    return 0;
}

static ConfigItem * ConfigParser_SeekItem(ConfigParser * me, const char * itemName, int * valPos)
{
    char *cptr;
    char *line;
    int len = strlen(itemName);
    ConfigSection * currentSection = me->currentSection;

    if (currentSection == NULL || len > MAX_CFG_LINE_LEN-1)
    {
#ifdef DEBUG
        printf("%s: Current Section is NULL\n", __func__);
#endif
        return NULL;
    }
    
#ifdef DEBUG
    printf("%s: Item: '%s'\n", __func__, itemName);
#endif

    ConfigItem * pItem = (ConfigItem *)LL_FirstEntry (&currentSection->items);
    for (;!LL_EndOfList(&currentSection->items, pItem); pItem = LL_NextEntry(pItem))
    {
        line = pItem->line;
        
        if(!strnicmp(line, itemName, len))	//found key
        {
            if (!isspace(line[len]) && (line[len] != me->delimiter))
            {
                continue;
            }
            
            if (me->delimiter)
            {
                cptr = &line[len];
                while(isspace(*cptr))	   //skip white space at before delimiter
                {
                    cptr++;
                }

                //find first delimiter
                if(*cptr != me->delimiter)
                {
                    continue; //invalid format
                }
                
                cptr++;                    //skip over delimiter

#ifdef DEBUG
                printf("%s: Item: '%s', Value: '%s'\n", __func__, itemName, cptr);
#endif
                *valPos = cptr - line;
            }
            else
            {
                *valPos = len + 1;
            }
            
            return pItem;
        }
    }
    
    return NULL;
}

char * ConfigParser_SeekItemGetValue(ConfigParser * me, const char * itemName)
{
    int valPos;
    ConfigItem * item = ConfigParser_SeekItem(me, itemName, &valPos);
    if (item == NULL)
    {
        return NULL;
    }

    char *cptr = &item->line[valPos];
    while(isspace(*cptr))	   //skip leading white space
    {
        cptr++;
    }

    return cptr;
}

int ConfigParser_ChangeItem(ConfigParser * me, const char * itemName, const char * val)
{
    int valPos;
    ConfigItem * item = ConfigParser_SeekItem(me, itemName, &valPos);
    if (item == NULL)
    {
        return 0;
    }

    char *pszValue = &item->line[valPos];
    int len = MAX_CFG_LINE_LEN - 1 - valPos;
    if (len > 0)
    {
        strncpy(pszValue, val, len);
        return 1;
    }

    return 0;
}

int ConfigParser_SeekItemGetChar(ConfigParser * me, const char * itemName, char * val)
{
    char *pszValue = ConfigParser_SeekItemGetValue(me, itemName);
    if (pszValue == NULL)
    {
        return 0;
    }

    *val = *pszValue;

    return 1;
}

int ConfigParser_SeekItemGetShort(ConfigParser * me, const char * itemName, short * val)
{
    long i;
    int ret = ConfigParser_SeekItemGetLong(me, itemName, &i);
    if (ret)
    {
        *val = (short)i;
    }
    
    return ret;
}

int ConfigParser_SeekItemGetWord(ConfigParser * me, const char * itemName, unsigned short * val)
{
    long i;
    int ret = ConfigParser_SeekItemGetLong(me, itemName, &i);
    if (ret)
    {
        *val = (unsigned short)i;
    }
    
    return ret;
}

int ConfigParser_SeekItemGetInt(ConfigParser * me, const char * itemName, int * val)
{
    long i;
    int ret = ConfigParser_SeekItemGetLong(me, itemName, &i);
    if (ret)
    {
        *val = (int)i;
    }

    return ret;
}

int ConfigParser_SeekItemGetLong(ConfigParser * me, const char * itemName, long * val)
{
    char *endptr;
    
    char *pszValue = ConfigParser_SeekItemGetValue(me, itemName);
    if (pszValue == NULL)
    {
        return 0;
    }

    long i = strtol(pszValue, &endptr, 0); /* Automatic base 10/16/8 switching */
    if (*pszValue != '\0' && (*endptr == '\0' || isspace(*endptr)))
    {
        *val = i;
    }
    else
    {
        return 0;
    }

    return 1;
}

int ConfigParser_SeekItemGetFloat(ConfigParser * me, const char * itemName, float * val)
{
    char *pszValue = ConfigParser_SeekItemGetValue(me, itemName);
    if (pszValue == NULL)
    {
        return 0;
    }
    
    *val = (float)atof(pszValue);
    
    return 1;
}

int ConfigParser_SeekItemGetDouble(ConfigParser * me, const char * itemName, double * val)
{
    char *pszValue = ConfigParser_SeekItemGetValue(me, itemName);
    if (pszValue == NULL)
    {
        return 0;
    }

    *val = atof(pszValue);

    return 1;
}


static int trim(char *s)
{
    size_t len = strlen(s);
    while (len > 0 && isspace(s[len-1]))
    {
        --len;
    }
    
    s[len] = '\0';

    return len;
}

int ConfigParser_AddItemString(ConfigParser * me, const char * itemName, const char * val)
{
    char buff[MAX_CFG_LINE_LEN];
    char ch = me->delimiter;
    ConfigSection * currentSection = me->currentSection;

    if (currentSection == NULL)
    {
        if (!LL_Empty( &me->sectionList ))
        {
            currentSection = (ConfigSection *)LL_FirstEntry (&me->sectionList);
        }
        else
        {
            return 0;
        }
    }

    if (ch == 0)
    {
        ch = ' ';
    }

    snprintf(buff, MAX_CFG_LINE_LEN - 1, "%s%c%s", itemName, ch, val);
    buff[MAX_CFG_LINE_LEN - 1] = '\0';

    if (!AddNewConfigItem(currentSection, buff))
    {
        return 0;
    }

    return 1;
}
