#include "load_and_parse_configuration_file.h"
#include "tw-ldm-interface-lib.h"
#include <twBaseTypes.h>
#include <stdio.h>
#include <cJSON.h>
#include <stdlib.h>
#include <string.h>

const char* PlcTagAccessMode_String_ReadOnly = "RD";
const char* PlcTagAccessMode_String_ReadWrite = "RDWR";
/* PLC Data type names, used to validate DataType field for tags. */
struct plc_data_type_
{
	const char * name;
	int		type;
};
typedef struct plc_data_type_ plc_data_type;

// Data types as defined in file lib/ocx/inc/ocxbpapi.h
const plc_data_type plc_data_types[] = 
{
	{"BOOL",	0xC1}, 
	{"SINT",	0xC2},
	{"INT",		0xC3},
	{"DINT",	0xC4},
	{"LINT",	0xC5},
	{"USINT", 0xC6},
	{"UINT",	0xC7},
	{"UDINT", 0xC8},
	{"ULINT", 0xC9},
	{"REAL",	0xCA},
	{"LREAL", 0xCB},
	{"BYTE",	0xD1},
	{"WORD",	0xD2},
	{"DWORD", 0xD3},
	{"LDOWRD",0xD4},
	{"STRING82",0x0FCE},
	{"NSTRING", 0x10001},
	{NULL, 0}
};

static int local_get_plc_data_type(const char* const data_type_name)
{
	int index = 0;

	if (!data_type_name)
		return 0;

	while (plc_data_types[index].name != NULL)
	{
		if (strcmp(plc_data_types[index].name, data_type_name) == 0)
			return plc_data_types[index].type;
		index++;
	}
	return 0;
}

static int local_parse_string(const cJSON* const parent_node, const char* const parent_node_name, const char* child_node_name, char** destination, size_t destination_length, int is_optional)
{
	int len;
	const cJSON* child_node;
	char* source = "";

	if (!parent_node || !parent_node_name || !child_node_name || !destination)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Function [local_parse_string] is called with invalid argument");
		return -1;
	}

	child_node = cJSON_GetObjectItem(parent_node, child_node_name);
	if (!child_node && !is_optional)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Field [%s] is missing on the node [%s]", child_node_name, parent_node_name);
		return -1;
	}
	
	if (!child_node)
	{
		if (!is_optional)
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Field [%s] is not defined on the node [%s]", child_node_name, parent_node_name);
			return -1;
		}
	}
	else if (!cJSON_IsString(child_node))
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Field [%s] has invalid type [%d] on the node [%s]", child_node_name, child_node->type, parent_node_name);
		return -1;
	}
	else if (!child_node->valuestring)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Field [%s] has NULL value on the node [%s]", child_node_name, parent_node_name);
		return -1;
	}
	else
	{
		source = child_node->valuestring;
	}

	len = strlen(source);
	if (!(*destination))
	{
		*destination = (char*) malloc(len + 1);
		if (!(*destination))
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to allocate memory for field [%s] of the node [%s], size = [%d] bytes", child_node_name, parent_node_name, len + 1);
			return -1;
		}
	}
	else
	{
		if (len >= (int) destination_length)
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Field [%s] of the node [%s] is too long: [%d] bytes, greater than buffer size [%d]", child_node_name, parent_node_name, len, destination_length);
			return -1;
		}
	}
	if (len > 0)
	{
		strncpy(*destination, source, len);
	}
	(*destination)[len] = 0;
	if (len < 1 && !is_optional)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Field [%s] is empty on the node [%s]", child_node_name, parent_node_name);
		return -1;
	}
	return 0;
}

int local_parse_integer(const cJSON* const parent_node, const char* const parent_node_name, const char* child_node_name, int32_t* destination, int is_optional, int32_t default_value)
{
	const cJSON* node;

	if (!parent_node || !parent_node_name || !child_node_name || !destination)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Function [local_parse_integer] is called with invalid argument(s)");
		return -1;
	}

	node = cJSON_GetObjectItem(parent_node, child_node_name);
	if (!node)
	{
		if (!is_optional)
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Node [%s] has no field [%s]", parent_node_name, child_node_name);
			return -1;
		}
		else
		{
			*destination = default_value;
			return 0;
		}
	}

	if (!cJSON_IsNumber(node))
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Field [%s] of the node [%s] has value of invalid type [%d]", child_node_name, parent_node_name, node->type);
		return -1;
	}
	*destination = node->valueint;
	return 0;
}

static void local_tw_ldm_free_aspects(tw_property_aspect* aspect)
{
	tw_property_aspect* current_aspect = aspect;
	while (current_aspect)
	{
		tw_property_aspect* next_aspect = current_aspect->next;
		if (current_aspect->name)
			free(current_aspect->name);
		free(current_aspect);
		current_aspect = next_aspect;
	}
}
static void local_tw_ldm_free_properties(tw_thing_property* property)
{
	tw_thing_property* current_property = property;
	while (current_property)
	{
		tw_thing_property* next_property = current_property->next;
		if (current_property->name)
			free(current_property->name);
		if (current_property->description)
			free(current_property->description);
		if (current_property->aspects)
			local_tw_ldm_free_aspects(current_property->aspects);
		free(current_property);
		current_property = next_property;
	}
}
static void local_tw_ldm_free_thing(tw_thing* thing)
{
	while (thing)
	{
		tw_thing* next_thing = thing->next;
		if (thing->name)
		{
			free(thing->name);
			thing->name = NULL;
		}
		if (thing->template_name)
		{
			free(thing->template_name);
			thing->template_name = NULL;
		}
		if (thing->is_connected_property_name)
		{
			free(thing->is_connected_property_name);
			thing->is_connected_property_name = NULL;
		}

		if (thing->properties)
		{
			local_tw_ldm_free_properties(thing->properties);
			thing->properties = NULL;
		}

		thing->next = NULL;
		free(thing);
		thing = next_thing;
	}

}

static void local_tw_ldm_free_tag(tw_tag* tag)
{
	tw_tag* current_tag = tag;
	tw_tag* next_tag = NULL;
	while (current_tag)
	{
		next_tag = current_tag->next;

		if (current_tag->tag_name)
			free(current_tag->tag_name);

		if (current_tag->thing)
			free(current_tag->thing);

		if (current_tag->property_name)
			free(current_tag->property_name);
		
		if (current_tag->tag_info)
			free(current_tag->tag_info);

		free(current_tag);
		current_tag = next_tag;
	}
}
void tw_ldm_free_config(tw_ldm_config* config)
{
	if (config->tw_server.app_key)
		free(config->tw_server.app_key);
	if (config->tw_server.hostname)
		free(config->tw_server.hostname);
	if (config->tw_server.path_to_root_ca_certificate_file)
		free(config->tw_server.path_to_root_ca_certificate_file);

	local_tw_ldm_free_thing(config->tw_server.things);
	local_tw_ldm_free_tag(config->tags);
	
	if (config->http_server.host)
		free(config->http_server.host);

	if (config->http_server.user_name)
		free(config->http_server.user_name);

	if (config->http_server.password)
		free(config->http_server.password);

	if (config->plc_path)
		free(config->plc_path);

	memset(config, 0, sizeof(*config));
}

int local_parse_tw_server_node(const cJSON* const root, tw_ldm_config* config)
{
	const cJSON* node;
	const cJSON* child_node;
	tw_thing* current_thing;
	tw_thing* previous_thing;
	cJSON* tw_server_node;
	char tmp_buffer[64];
	int32_t tmp_int32;

	int result = -1;

	if (!root || !config)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Function [local_parse_tw_server_node] is called with invalid argument(s)");
		return -1;
	}
	config->tw_server.polling_rate = 1000;

	tw_server_node = cJSON_GetObjectItem(root, "TwServer");
	if (!tw_server_node)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Could not find node[TwServer] in the configuration file");
		return -1;
	}
	if (!cJSON_IsObject(tw_server_node))
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Node[TwServer] has invalid value in the configuration file");
		return -1;
	}

	/* Parse ThingWorx host name */
	result = local_parse_string(tw_server_node, "TwServer", "Host", &config->tw_server.hostname, 0, 0);
	if (result != 0)
		return result;

	result = local_parse_string(tw_server_node, "TwServer", "RootCaFileName", &config->tw_server.path_to_root_ca_certificate_file, 0, 1);
	if (result != 0)
		return result;

	/* Parse ThingWorx port number */
	result = local_parse_integer(tw_server_node, "TwServer", "Port", &tmp_int32, 1, 443);
	if (result != 0)
		return result;
	config->tw_server.port = tmp_int32;

	if (config->tw_server.port <= 0)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Node [TwServer/Port] has invalid value");
		return -1;
	}

	result = local_parse_integer(tw_server_node, "TwServer", "UseHttps", &tmp_int32, 0, 0);
	if (result == 0)
		config->tw_server.use_https = tmp_int32;
	else
	{
		/* If not defined, use default value, based on port number  */
		if (config->tw_server.port == 80 || config->tw_server.port == 8080)
			config->tw_server.use_https = 0;
		else
			config->tw_server.use_https = 1;
	}

	result = local_parse_integer(tw_server_node, "TwServer", "DisableCertificateValidation", &tmp_int32, 0, 0);
	if (result == 0)
		config->tw_server.disable_certificate_validation = tmp_int32;
	else
	{
		/* Not defined, use default value: do not disable validation */
		config->tw_server.disable_certificate_validation = 0;
	}

	if ((config->tw_server.port == 443 || config->tw_server.port == 8443) && !config->tw_server.use_https)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Node [TwServer/UseHttps] has invalid value [%d], incompatible with port number [%d]", config->tw_server.use_https, config->tw_server.port);
		return -1;
	}

	result = local_parse_integer(tw_server_node, "TwServer", "Timeout", &tmp_int32, 1, 5000);
	if (result != 0)
		return result;
	config->tw_server.timeout = (uint32_t)tmp_int32;
	if (config->tw_server.timeout < 500)
		config->tw_server.timeout = 500;
	else if (config->tw_server.timeout > 60000)
		config->tw_server.timeout = 60000;

	result = local_parse_integer(tw_server_node, "TwServer", "ConnectionRetryCount", &tmp_int32, 1, 0x7fff);
	if (result != 0)
		return result;
	config->tw_server.connection_retry_count = (uint32_t)tmp_int32;
	if (config->tw_server.connection_retry_count < 1)
		config->tw_server.connection_retry_count = 1;

	/* Parse ThingWorx application key (if defined)*/
	result = local_parse_string(tw_server_node, "TwServer/AppKey", "AppKey", &config->tw_server.app_key, 0, 0);
	if (result != 0)
		return result;

	node = cJSON_GetObjectItem(tw_server_node, "Things");
	if (!node || !cJSON_IsArray(node))
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Node [Things] is not defined or have invalid type");
		return -1;
	}

	previous_thing = NULL;
	cJSON_ArrayForEach(child_node, node)
	{
		cJSON* properties_node;
		cJSON* property_node;
		tw_thing_property* previous_property = NULL;

		current_thing = (tw_thing*)malloc(sizeof(tw_thing));
		if (!current_thing)
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Out of memory");
			return -1;
		}
		memset(current_thing, 0, sizeof(*current_thing));
		if (previous_thing)
			previous_thing->next = current_thing;
		
		if (!config->tw_server.things)
			config->tw_server.things = current_thing;

		/* Read Name */
		result = local_parse_string(child_node, "Thing", "Name", &current_thing->name, 0, 0);
		if (result != 0)
			return result;

		/* Read Template name */
		result = local_parse_string(child_node, "Thing", "Template", &current_thing->template_name, 0, 0);
		if (result != 0)
			return result;
		
		local_parse_string(child_node, "Thing", "IsConnectedPropertyName", &current_thing->is_connected_property_name, 0, 1);
		if (!current_thing->is_connected_property_name)
			tw_ldm_log(TW_LDM_LOG_NOTICE, "Thing [%s] does not have property to show if PLC is connected.", current_thing->name);
		else
			tw_ldm_log(TW_LDM_LOG_NOTICE, "Thing [%s] has property [%s] to show if PLC is connected.", current_thing->name, current_thing->is_connected_property_name);

		properties_node = cJSON_GetObjectItem(child_node, "Properties");
		if (!properties_node)
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Could not find node [Properties] for the thing node [%s]", current_thing->name);
			return -1;
		}

		if (!cJSON_IsArray(properties_node))
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Node [Properties] for the thing node [%s] has invalid type [%d]", current_thing->name, properties_node->type);
			return -1;
		}

		cJSON_ArrayForEach(property_node, properties_node)
		{
			tw_thing_property* property = (tw_thing_property*)malloc(sizeof(tw_thing_property));
			char* p_buffer = &tmp_buffer[0];
			result = -1;

			if (!property)
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to allocate memory for properties of the thing [%s]", current_thing->name);
				return -1;
			}
			memset(property, 0, sizeof(*property));

			if (!current_thing->properties)
				current_thing->properties = property;
			if (previous_property)
			{
				previous_property->next = property;
			}

			// Now parse fields for this particular property:
			if (!cJSON_IsObject(property_node))
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "Thing [%s] has property node of invalid type [%d]", current_thing->name, property_node->type);
				return -1;
			}
			result = local_parse_string(property_node, current_thing->name, "Name", &property->name, 0, 0);
			if (result != 0)
				return result;
			result = local_parse_string(property_node, current_thing->name, "Description", &property->description, 0, 1);
			if (result != 0)
				return result;
			
			memset(tmp_buffer, 0, sizeof(tmp_buffer));
			result = local_parse_string(property_node, current_thing->name, "Type", &p_buffer, sizeof(tmp_buffer) - 1, 0);
			if (result != 0)
				return result;
			property->type = baseTypeFromString(tmp_buffer);
			if (property->type == TW_NOTHING || property->type == TW_UNKNOWN_TYPE)
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "Thing [%s] has property node with invalid data type [%s]", current_thing->name, tmp_buffer);
				return -1;
			}
			// TODO: aspects.

			previous_property = property;
			result = 0;
		}

		if (result != 0)
			return result;

		previous_thing = current_thing;
	}

	return 0;
}

static int local_parse_http_server_node(const cJSON* const root, tw_ldm_config* config)
{
	cJSON* http_server_node;

	int result = -1;

	if (!root || !config)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Function [local_parse_tw_server_node] is called with invalid argument(s)");
		return -1;
	}

	http_server_node = cJSON_GetObjectItem(root, "HttpServer");
	if (!http_server_node)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Could not find node [HttpServer] in the configuration file");
		return -1;
	}
	if (!cJSON_IsObject(http_server_node))
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Node [HttpServer] has invalid type [%d] in the configuration file", http_server_node->type);
		return -1;
	}

	result = local_parse_string(http_server_node, "HttpServer", "Host", &config->http_server.host, 0, 0);
	if (result != 0)
		return result;

	result = local_parse_integer(http_server_node, "HttpServer", "Port", &config->http_server.port, 1, 8080);
	if (result != 0)
		return result;

	/* TODO: USer name and password are marked as optional for now. Change when they become required */
	result = local_parse_string(http_server_node, "HttpServer", "UserName", &config->http_server.user_name, 0, 0);
	if (result != 0)
		return result;

	result = local_parse_string(http_server_node, "HttpServer", "Password", &config->http_server.password, 0, 0);
	if (result != 0)
		return result;

	return 0;
}

int local_validate_thing_name_and_property(const tw_ldm_config* config, const char* const tag_name, const char* const thing_name, const char* const property_name)
{
	const tw_thing* current_thing = config->tw_server.things;
	int thing_found = 0;
	if (!config || !tag_name || !thing_name || !property_name)
		return -1;
	while (current_thing)
	{
		if (strcmp(current_thing->name, thing_name) == 0)
		{
			thing_found = 1;
			tw_thing_property* current_property = current_thing->properties;
			while (current_property)
			{
				if (strcmp(property_name, current_property->name) == 0)
				{
					return 0;
				}
				current_property = current_property->next;
			}
			break;
		}
		current_thing = current_thing->next;
	}
	if (!thing_found)
		tw_ldm_log(TW_LDM_LOG_ERROR, "Tag [%s]: Thing with name [%s] not found", tag_name, thing_name);
	else
		tw_ldm_log(TW_LDM_LOG_ERROR, "Tag [%s]: Thing with name [%s] does not have property [%s]", tag_name, thing_name, property_name);

	return -1;
}

int local_parse_tags(const cJSON* const root, tw_ldm_config* config)
{
	tw_tag* previous_tag = NULL;
	cJSON* tags_node;
	cJSON* tag_node;
	int result = -1;

	if (!root || !config)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Function [local_parse_tags] is called with invalid argument(s)");
		return -1;
	}

	tags_node = cJSON_GetObjectItem(root, "Tags");
	if (!tags_node)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Could not find node [Tags] in the configuration file");
		return -1;
	}
	if (!cJSON_IsArray(tags_node))
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Node [Tags] has invalid type [%d] in the configuration file", tags_node->type);
		return -1;
	}

	cJSON_ArrayForEach(tag_node, tags_node)
	{
		tw_tag* tag = (tw_tag*) malloc(sizeof(tw_tag));
		char* tmp_value = NULL;
		char tmp_buffer[32];
		int offset = 0;

		if (!tag)
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to allocate memory for a tag");
			return -1;
		}
		memset(tag, 0, sizeof(*tag));
		memset(tmp_buffer, 0, sizeof(tmp_buffer));

		if (!config->tags)
			config->tags = tag;
		if (previous_tag)
			previous_tag->next = tag;

		// Now parse fields for this particular tag:
		if (!cJSON_IsObject(tag_node))
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Configuration has tag node of invalid type [%d]", tag_node->type);
			return -1;
		}

		// Read as optional:
		result = local_parse_string(tag_node, "Tags", "Tag", &tag->tag_name, 0, 1);
		if (result != 0)
			return result;
		
		// Read as mandatory (to know if it was present there):
		result = local_parse_integer(tag_node, "Tags", "Offset", &offset, 0, 0);
		tag->offset = (uint16_t) offset;
		if (tag->tag_name && strlen(tag->tag_name) > 0)
		{
			// Tag name is defined.
			tag->use_offset = FALSE;
		}
		else
		{
			tag->use_offset = TRUE;

			if (result == 0)
			{
				// Offset was parsed successfully,so it is defined.
				// Compose tag_name to use for diagnostics purposes:
				int str_length;
				sprintf(tmp_buffer, "Offset %d", tag->offset);
				str_length = strlen(tmp_buffer);
				tag->tag_name = (char*)malloc(str_length + 1);
				if (tag->tag_name)
				{
					memcpy(tag->tag_name, tmp_buffer, str_length);
					tag->tag_name[str_length] = 0;
				}
				else
				{
					tw_ldm_log(TW_LDM_LOG_ERROR, "Out of memory error occurred during parsing of tag [%s]", tmp_buffer);
					result = -1;
				}
			}
			else
			{
				return result;
			}
		}

		if (result == 0)
		{
			result = local_parse_integer(tag_node, "Tags", "ScanRate", &tag->scan_rate, 0, 0);
			if (tag->scan_rate == 0)
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "Tag [%s] has invalid scan rate (must be greater than 0)", tag->tag_name);
				result = -1;
			}
		}
		result = local_parse_string(tag_node, "Tags", "DataType", &tmp_value , 0, 0);
		if (result != 0)
			return result;
		tag->data_type = local_get_plc_data_type(tmp_value);
		if (tmp_value)
		{
			free(tmp_value);
			tmp_value = NULL;
		}
		if (tag->data_type == 0)
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Tag [%s] has invalid data type", tag->tag_name);
			return -1;
		}

		result = local_parse_integer(tag_node, "Tags", "ScanRate", &tag->scan_rate, 0, 0);
		if (result != 0)
			return result;
		if (tag->scan_rate == 0)
		{
			tw_ldm_log(TW_LDM_LOG_ERROR, "Tag [%s] has invalid scan rate (must be greater than 0)", tag->tag_name);
			result = -1;
		}

		if (result == 0)
			result = local_parse_string(tag_node, "Tags", "Access", &tmp_value, 0, 0);

		if (result == 0)
		{
			int len = strlen(tmp_value);
			int len_r = strlen(PlcTagAccessMode_String_ReadOnly);
			int len_rw = strlen(PlcTagAccessMode_String_ReadWrite);

			if ( (len == len_rw) && (memcmp(tmp_value, PlcTagAccessMode_String_ReadWrite, len_rw) == 0))
				tag->access_mode = PlcTagAccessMode_ReadWrite;
			else if ( (len == len_r) && (memcmp(tmp_value, PlcTagAccessMode_String_ReadOnly, len_r) == 0) )
				tag->access_mode = PlcTagAccessMode_ReadOnly;
			else
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "Tag [%s] has invalid access mode [%s]", tag->tag_name, tmp_value);
				result = -1;
			}
		}
		if (result == 0)
			result = local_parse_string(tag_node, tag->tag_name, "Thing", &tag->thing, 0, 0);
		if (result == 0)
			result = local_parse_string(tag_node, "Tags", "Property", &tag->property_name, 0, 0);
		if (result == 0)
			result = local_validate_thing_name_and_property(config, tag->tag_name, tag->thing, tag->property_name);

		if (tmp_value)
			free(tmp_value);

		if (result != 0)
			return result;
		if (tag->scan_rate < config->tw_server.polling_rate)
			config->tw_server.polling_rate = tag->scan_rate;

		previous_tag = tag;
	}

	return 0;
}

int load_and_parse_configuration_file(const char* path_to_file, tw_ldm_config* config)
{
	FILE* fp;
	long file_size;
	char* file_content;
	size_t read_size;
	cJSON* root;
	int result = -1;
	int tmp_int;

	if (!path_to_file)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Invalid argument [path_to_file] in function [load_and_parse_configuraiton_file]");
		return -1;
	}

	if (!config)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Invalid argument [config] in function [load_and_parse_configuraiton_file]");
		return -1;
	}

	fp = fopen(path_to_file, "rb");
	if (!fp)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to open configuration file [%s] to read", path_to_file);
		return -1;
	}

	if (fseek(fp, 0L, SEEK_END) != 0)
	{
		fclose(fp);
		tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to determine size of the configuration file [%s]", path_to_file);
		return -1;
	}
	file_size = ftell(fp);
	rewind(fp);

	file_content = malloc(file_size + 1);
	if (!file_content)
	{
		fclose(fp);
		tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to allocate %d bytes to read context of the configuration file [%s]", file_size + 1, path_to_file);
		return -1;
	}
	memset(file_content, 0, file_size + 1);
	read_size = fread(file_content, 1, file_size, fp);
	if (fclose(fp) != 0)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Closing of the configuration file [%s] was not successful", path_to_file);
	}

	if (read_size != file_size)
	{
		free(file_content);
		tw_ldm_log(TW_LDM_LOG_ERROR, "Number of bytes [%d] read from file [%s] is not as expected [%d]", read_size, path_to_file, file_size);
		return -1;
	}

	root = cJSON_Parse(file_content);
	free(file_content);

	if (root == NULL)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "Filed to parse configuration file [%s]", path_to_file);
		return -1;
	}

	memset(config, 0, sizeof(*config));

	result = local_parse_tw_server_node(root, config);

	// HttpServer node is not used currently. TODO: uncomment when it becomes required.
	//if (result == 0)
	//	result = local_parse_http_server_node(root, config);

	if (result == 0)
		result = local_parse_string(root, "Configuration file", "PlcPath", &config->plc_path, 0, 0);

	if (result == 0)
		local_parse_integer(root, "Configuration file", "SyncTimeWithPlc", &config->sync_time_with_plc, 1, 0);

	tmp_int = 0;

	if (result == 0)
		result = local_parse_integer(root, "Configuration file", "PlcWriteOffset", &tmp_int, 1, 0);
	config->plc_write_offset = (uint16_t)tmp_int;

	if (result == 0)
		result = local_parse_integer(root, "Configuration file", "StatusPrintIntervalInSeconds", &config->status_print_interval, 1, 10);
	if (config->status_print_interval == 0)
		config->status_print_interval = 10;

	if (result == 0)
		result = local_parse_tags(root, config);
	if (result == 0)
		result = local_parse_integer(root, "Configuration file", "UpdateTagInfoBeforeEachOperation", &config->get_tag_info_before_each_op, 1, 1);

	if (root)
		cJSON_Delete(root);

	return result;
}