/////////////////////////////////////////////////////////////////////////////
//
// wattcpedit.c
//
// Author     : Iaki Zuloaga Izaguirre, Prosoft Technology Inc
// Date       : 08/07/2008
// Description: This file allows to read and change the wattcp
//              network configuration file.
//
//
// Modified:
//  08/07/08 -- File Created
//  08/27/08 -- IZI Added logic to save and load config date from file
//  12/03/2008 RAR Altered code slightly for return values, formatting and
//                 other things.
//  11/01/2011 HYU - Added saveWattcpNetworkData().
//  10/17/2012 HYU - Added saveWattcpNetworkDataEx().
//  11/27/2012 JOE - Added getWattcpNetworkDataEx().
//
/////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <strings.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/stat.h>
#include "nethelper.h"
#include "wattcpedit.h"
#include "cfgparse.h"


/////////////////////////////////////////////////////////////////
// Definitions
/////////////////////////////////////////////////////////////////
#define FIELD_SIZE 100

#define MODE_NAME 0
#define MODE_VALUE 1
#define MODE_COMMENT 2


/******************************************************************************
* Purpose: This function stores network data (IP, netmask and gateway) into
*          a byte array.
*
* Params: char *buff: buffer where the data is going to be stored using one
*                     integer for each address element and in the order:
*                     IP, NETMASK, GATEWAY
*
*         int *buff_len: The total size of the buffer is returned (always 24)
*
* 	       int *ip: pointer to int[4] to store the ip address.
*
*         int *netmask: pointer to int[4] to store the network mask.
*
*         int *gateway: pointer to int[4] to store the gateway.
*
* Returns: Nothing
*
* Comments:
*
*****************************************************************************/
void networkDataToByteBuffer(char * buff, int *  buff_len, int *ip, int *netmask, int *gateway)
{
    short *intBuff = (short *)buff;
    
    intBuff[0]  = ip[0];
    intBuff[1]  = ip[1];
    intBuff[2]  = ip[2];
    intBuff[3]  = ip[3];
    intBuff[4]  = netmask[0];
    intBuff[5]  = netmask[1];
    intBuff[6]  = netmask[2];
    intBuff[7]  = netmask[3];
    intBuff[8]  = gateway[0];
    intBuff[9]  = gateway[1];
    intBuff[10] = gateway[2];
    intBuff[11] = gateway[3];
    *buff_len   = 24;
}


/******************************************************************************
* Purpose: This function stores network data (IP, netmask and gateway) into
*          user variables from array.
*
* Params: char *buff: buffer where the data is going to be read using one
*                     integer for each address element and in the order:
*                     IP, NETMASK, GATEWAY
*
* 	       int *ip: pointer to int[4] to store the ip address.
*
*         int *netmask: pointer to int[4] to store the network mask.
*
*         int *gateway: pointer to int[4] to store the gateway.
*
* Returns: Nothing
*
* Comments:
*
*****************************************************************************/
void byteBufferToNetworkData(char * buff, int *ip, int *netmask, int *gateway)
{
    short *intBuff = (short *)buff;
    
    ip[0]      = intBuff[0];
    ip[1]      = intBuff[1];
    ip[2]      = intBuff[2];
    ip[3]      = intBuff[3];
    netmask[0] = intBuff[4];
    netmask[1] = intBuff[5];
    netmask[2] = intBuff[6];
    netmask[3] = intBuff[7];
    gateway[0] = intBuff[8];
    gateway[1] = intBuff[9];
    gateway[2] = intBuff[10];
    gateway[3] = intBuff[11];
}


/******************************************************************************
* Purpose: This function parses a comment line and returns the Last Change
*          date stored in comments
*
* Params: char *commentLine: Is the string that has the comment of the
*                            configuration file
*
*         struct tm *configDateTime: returned time from the comment line
*
* Returns: Error Code: 0 OK
*                      -1 error or no Last Change in the comment line
*
* Comments:
*
*****************************************************************************/
int getConfigDateTime(char *commentLine, struct tm *configDateTime)
{
    char *dateString;
    int result;
    const char *LAST_CHANGE_STRING = "Last Change: ";
    struct tm readDateTime;
    readDateTime = *configDateTime;
    
    dateString = strstr(commentLine,LAST_CHANGE_STRING);
    if (dateString != NULL)
    {
        // We have to add the length of "Last Change: "
        dateString+=strlen(LAST_CHANGE_STRING);
        result = sscanf(dateString,"%d.%d.%d %d:%d",&readDateTime.tm_year,
            &readDateTime.tm_mon,
            &readDateTime.tm_mday,
            &readDateTime.tm_hour,
            &readDateTime.tm_min);
        if (result == 5)
        {
            *configDateTime = readDateTime;
            //printf("read Correctly:\n");
            //printf("year: %d, month: %d, day: %d\n",configDateTime->tm_year,configDateTime->tm_mon,configDateTime->tm_mday);
            //printf("Hour: %d, Minute: %d\n",configDateTime->tm_hour,configDateTime->tm_min);
            return NETHELPER_EXIT_SUCCESS;
        }
    }
    return NETHELPER_EXIT_FAILURE;
}


/******************************************************************************
* Purpose: This function returns the information stored in the wattcp.cfg file.
*
* Params: int *ip: pointer to int[4] to store the ip address. On error will
*                  return -1 in all the fields
*
*         int *netmask: pointer to int[4] to store the network mask. On error will
*                       return -1 in all the fields
*
*         int *gateway: pointer to int[4] to store the gateway. On error will
*                       return -1 in all the fields
*
*         struct tm *configDateTime: pointer to last cfg date/time. On error
*                       will return 0 in fields, except tm_yday set to -1.
*
* Returns: Error Code: 0x1B Cannot open file
*
* Comments: If the function was not able to find the date on the file it sets
*           -1 in the tm_yday field of configDateTime
*
*****************************************************************************/
unsigned char getWattcpNetworkData(int *ip, int *netmask, int *gateway, struct tm *configDateTime)
{
    FILE *fptr;
    char filedata[500];
    char *currptr;
    int mode;
    char name[FIELD_SIZE+1];
    char value[FIELD_SIZE+1];
    int nameIndex;
    int valueIndex;
    struct in_addr inAddr;
    unsigned char *ipArray = (unsigned char*)&inAddr;
    struct tm configDateTimeCache;
    
    // on error return -1 in all fields, so all fields are initialized
    // and if we succesfuly get the values then we put the right values
    if (ip != NULL)
    { ip[0] = -1; ip[1] = -1; ip[2] = -1; ip[3] = -1; }
    
    if (netmask != NULL)
    { netmask[0] = -1; netmask[1] = -1; netmask[2] = -1; netmask[3] = -1;}
    
    if (gateway != NULL)
    {gateway[0] = -1; gateway[1] = -1; gateway[2] = -1; gateway[3] = -1;}
    
    if (configDateTime != NULL)
    {
        configDateTime->tm_year = 0;
        configDateTime->tm_mon  = 0;
        configDateTime->tm_mday = 0;
        configDateTime->tm_hour = 0;
        configDateTime->tm_min  = 0;
        configDateTime->tm_yday = -1;
    }
    
    if ((fptr = fopen(WATTCP_CFG_FILENAME, "r")) == NULL)
    {
        return 0x1B;
    }
    
    // Read all the wattcp file
    while (fgets(filedata, 500, fptr) != NULL)
    {
        currptr	  = filedata;
        name[0]    = 0;
        value[0]   = 0;
        nameIndex  = 0;
        valueIndex = 0;
        mode       = MODE_NAME;
        
        // Continue until the string ends
        while (*currptr!=0)
        {
            //printf("currptr = '%c' ",*currptr);
            switch (*currptr)
            {
            case '\n':
            case '\r':
            case ' ':
            case '\t':
                break;
                
            case '#':
            case ';':
                mode = MODE_COMMENT;
                break;
                
            case '=':
                if (mode == MODE_NAME)
                {
                    mode = MODE_VALUE;
                }
                break;
                
            default:
                if (mode == MODE_NAME)
                {
                    //printf("adding to name\n");
                    name[nameIndex] = *currptr;
                    nameIndex++;
                    if (nameIndex >= FIELD_SIZE)
                    {
                        nameIndex = FIELD_SIZE;
                    }
                    name[nameIndex] = 0;
                }
                if (mode == MODE_VALUE)
                {
                    //printf("adding to value\n");
                    value[valueIndex] = *currptr;
                    valueIndex++;
                    if (valueIndex >= FIELD_SIZE)
                    {
                        valueIndex = FIELD_SIZE;
                    }
                    value[valueIndex] = 0;
                }
                
            }
            // If finished parsing, break the loop
            // but analyze comments to extract date and time information
            if (mode == MODE_COMMENT)
            {
                if (configDateTime != NULL)
                {	// Only if we got the date and time right we return that to the user
                    // This is to avoid parsing errors
                    if (getConfigDateTime(currptr, &configDateTimeCache) == NETHELPER_EXIT_SUCCESS)
                    {
                        *configDateTime = configDateTimeCache;
                    }
                }
                break;
            }
            currptr++;
        }
        
        if ((ip != NULL) && (strcasecmp(name, "my_ip") == 0))
        {
            //printf("MY_IP received, IP=%s\n",value);
            if (inet_aton(value, &inAddr) != 0)
            {
                ip[0] = ipArray[0];
                ip[1] = ipArray[1];
                ip[2] = ipArray[2];
                ip[3] = ipArray[3];
            }
        }
        if ((netmask != NULL) && (strcasecmp(name, "netmask") == 0))
        {
            //printf("MY_IP received, IP=%s\n",value);
            if (inet_aton(value,&inAddr) != 0)
            {
                netmask[0] = ipArray[0];
                netmask[1] = ipArray[1];
                netmask[2] = ipArray[2];
                netmask[3] = ipArray[3];
            }
        }
        if ((gateway != NULL) && (strcasecmp(name, "gateway") == 0))
        {
            //printf("MY_IP received, IP=%s\n",value);
            if (inet_aton(value,&inAddr) != 0)
            {
                gateway[0] = ipArray[0];
                gateway[1] = ipArray[1];
                gateway[2] = ipArray[2];
                gateway[3] = ipArray[3];
            }
        }
    }
    fclose(fptr);
    return NETHELPER_EXIT_SUCCESS;
}


/******************************************************************************
* Purpose: This function returns the information stored in the wattcp.cfg file.
*
* Params: int *ip: pointer to int[4] to store the ip address. On error will
*                  return -1 in all the fields
*
*         int *netmask: pointer to int[4] to store the network mask. On error will
*                       return -1 in all the fields
*
*         int *gateway: pointer to int[4] to store the gateway. On error will
*                       return -1 in all the fields
*
*         struct tm *configDateTime: pointer to last cfg date/time. On error
*                       will return 0 in fields, except tm_yday set to -1.
*
*         int type: integer to get the type of communication configuration. On error will
*                       return -1 in all the fields
*
* Returns: Error Code: 0x1B Cannot open file
*
* Comments: If the function was not able to find the date on the file it sets
*           -1 in the tm_yday field of configDateTime
*
*****************************************************************************/
unsigned char getWattcpNetworkDataEx(int *ip, int *netmask, int *gateway, struct tm *configDateTime, int type)
{
    FILE *fptr;
    char filedata[500];
    char *currptr;
    int mode;
    char name[FIELD_SIZE+1];
    char value[FIELD_SIZE+1];
    int nameIndex;
    int valueIndex;
    struct in_addr inAddr;
    unsigned char *ipArray = (unsigned char*)&inAddr;
    struct tm configDateTimeCache;
    
    // on error return -1 in all fields, so all fields are initialized
    // and if we succesfuly get the values then we put the right values
    if (ip != NULL)
    { ip[0] = -1; ip[1] = -1; ip[2] = -1; ip[3] = -1; }
    
    if (netmask != NULL)
    { netmask[0] = -1; netmask[1] = -1; netmask[2] = -1; netmask[3] = -1;}
    
    if (gateway != NULL)
    {gateway[0] = -1; gateway[1] = -1; gateway[2] = -1; gateway[3] = -1;}
    
    if (configDateTime != NULL)
    {
        configDateTime->tm_year = 0;
        configDateTime->tm_mon  = 0;
        configDateTime->tm_mday = 0;
        configDateTime->tm_hour = 0;
        configDateTime->tm_min  = 0;
        configDateTime->tm_yday = -1;
    }
    
    if ((fptr = fopen(WATTCP_CFG_FILENAME, "r")) == NULL)
    {
        return 0x1B;
    }
    
    // Read all the wattcp file
    while (fgets(filedata, 500, fptr) != NULL)
    {
        currptr	  = filedata;
        name[0]    = 0;
        value[0]   = 0;
        nameIndex  = 0;
        valueIndex = 0;
        mode       = MODE_NAME;
        
        // Continue until the string ends
        while (*currptr!=0)
        {
            //printf("currptr = '%c' ",*currptr);
            switch (*currptr)
            {
            case '\n':
            case '\r':
            case ' ':
            case '\t':
                break;
                
            case '#':
            case ';':
                mode = MODE_COMMENT;
                break;
                
            case '=':
                if (mode == MODE_NAME)
                {
                    mode = MODE_VALUE;
                }
                break;
                
            default:
                if (mode == MODE_NAME)
                {
                    //printf("adding to name\n");
                    name[nameIndex] = *currptr;
                    nameIndex++;
                    if (nameIndex >= FIELD_SIZE)
                    {
                        nameIndex = FIELD_SIZE;
                    }
                    name[nameIndex] = 0;
                }
                if (mode == MODE_VALUE)
                {
                    //printf("adding to value\n");
                    value[valueIndex] = *currptr;
                    valueIndex++;
                    if (valueIndex >= FIELD_SIZE)
                    {
                        valueIndex = FIELD_SIZE;
                    }
                    value[valueIndex] = 0;
                }
                
            }
            // If finished parsing, break the loop
            // but analyze comments to extract date and time information
            if (mode == MODE_COMMENT)
            {
                if (configDateTime != NULL)
                {	// Only if we got the date and time right we return that to the user
                    // This is to avoid parsing errors
                    if (getConfigDateTime(currptr, &configDateTimeCache) == NETHELPER_EXIT_SUCCESS)
                    {
                        *configDateTime = configDateTimeCache;
                    }
                }
                break;
            }
            currptr++;
        }
        
        switch (type)
        {
           case IP_CFG_PORT1:
           default:
              if ((ip != NULL) && (strcasecmp(name, "my_ip") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value, &inAddr) != 0)
                  {
                      ip[0] = ipArray[0];
                      ip[1] = ipArray[1];
                      ip[2] = ipArray[2];
                      ip[3] = ipArray[3];
                  }
              }
              if ((netmask != NULL) && (strcasecmp(name, "netmask") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value,&inAddr) != 0)
                  {
                      netmask[0] = ipArray[0];
                      netmask[1] = ipArray[1];
                      netmask[2] = ipArray[2];
                      netmask[3] = ipArray[3];
                  }
              }
              if ((gateway != NULL) && (strcasecmp(name, "gateway") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value,&inAddr) != 0)
                  {
                      gateway[0] = ipArray[0];
                      gateway[1] = ipArray[1];
                      gateway[2] = ipArray[2];
                      gateway[3] = ipArray[3];
                  }
              }
        	     break;

           case IP_CFG_PORT2:
              if ((ip != NULL) && (strcasecmp(name, "my_ip_1") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value, &inAddr) != 0)
                  {
                      ip[0] = ipArray[0];
                      ip[1] = ipArray[1];
                      ip[2] = ipArray[2];
                      ip[3] = ipArray[3];
                  }
              }
              if ((netmask != NULL) && (strcasecmp(name, "netmask_1") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value,&inAddr) != 0)
                  {
                      netmask[0] = ipArray[0];
                      netmask[1] = ipArray[1];
                      netmask[2] = ipArray[2];
                      netmask[3] = ipArray[3];
                  }
              }
              if ((gateway != NULL) && (strcasecmp(name, "gateway_1") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value,&inAddr) != 0)
                  {
                      gateway[0] = ipArray[0];
                      gateway[1] = ipArray[1];
                      gateway[2] = ipArray[2];
                      gateway[3] = ipArray[3];
                  }
              }
              break;

           case IP_CFG_WLAN:
              if ((ip != NULL) && (strcasecmp(name, "Radio_IP") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value, &inAddr) != 0)
                  {
                      ip[0] = ipArray[0];
                      ip[1] = ipArray[1];
                      ip[2] = ipArray[2];
                      ip[3] = ipArray[3];
                  }
              }
              if ((netmask != NULL) && (strcasecmp(name, "Radio_Subnet_Mask") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value,&inAddr) != 0)
                  {
                      netmask[0] = ipArray[0];
                      netmask[1] = ipArray[1];
                      netmask[2] = ipArray[2];
                      netmask[3] = ipArray[3];
                  }
              }
              if ((gateway != NULL) && (strcasecmp(name, "Radio_Default_Gateway") == 0))
              {
                  //printf("MY_IP received, IP=%s\n",value);
                  if (inet_aton(value,&inAddr) != 0)
                  {
                      gateway[0] = ipArray[0];
                      gateway[1] = ipArray[1];
                      gateway[2] = ipArray[2];
                      gateway[3] = ipArray[3];
                  }
              }
              break;
        }
    }
    fclose(fptr);
    return NETHELPER_EXIT_SUCCESS;
}

int validateIpConfigData(IpConfigData * configData)
{
    // make sure all the fields are OK
    if ((configData->ip[0] < 0)      || (configData->ip[0] > 255) ||
        (configData->ip[1] < 0)      || (configData->ip[1] > 255) ||
        (configData->ip[2] < 0)      || (configData->ip[2] > 255) ||
        (configData->ip[3] < 0)      || (configData->ip[3] > 255))
    {
        return -1;
    }
    
    if ((configData->options & HAS_IP_NETMASK) &&
        ((configData->netmask[0] < 0) || (configData->netmask[0] > 255) ||
        (configData->netmask[1] < 0) || (configData->netmask[1] > 255) ||
        (configData->netmask[2] < 0) || (configData->netmask[2] > 255) ||
        (configData->netmask[3] < 0) || (configData->netmask[3] > 255)))
    {
        return -2;
    }
    
    if ((configData->options & HAS_IP_GATEWAY) &&
        ((configData->gateway[0] < 0) || (configData->gateway[0] > 255) ||
        (configData->gateway[1] < 0) || (configData->gateway[1] > 255) ||
        (configData->gateway[2] < 0) || (configData->gateway[2] > 255) ||
        (configData->gateway[3] < 0) || (configData->gateway[3] > 255)))
    {
        return -3;
    }
    
    return 0;
}

/******************************************************************************
* Purpose: This function creates a new wattcp file with the selected onfiguration
*
* Params: int *ip: pointer to int[4] that stores the new ip address.
*
*         int *netmask: pointer to int[4] that stores the network mask.
*
*         int *gateway: pointer to int[4] that stores the gateway.
*
* Returns: Error Code: 0x30 Invalid Setting
*                      0x1B Cannot open file
*
* Comments:
*
*****************************************************************************/
unsigned char setWattcpNetworkData(IpConfigData * configData, int numConfig)
{
    int i;
    FILE *fptr;
    int fd;
    
    // make sure all the fields are OK
    for (i = 0; i < numConfig; i++)
    {
        if (validateIpConfigData(&configData[i]) < 0)
        {
            return 0x30;
        };
    }
    
    // Get current time
    time_t currentTime;
    struct tm *localTime;
    time(&currentTime);
    localTime = localtime(&currentTime);
    
    if ((fptr = fopen(WATTCP_CFG_FILENAME, "w")) == NULL)
    {
        return 0x1B;
    }
    
    fprintf(fptr,"# ProSoft Technology Inc.\r\n");
    fprintf(fptr,"# Automatically generated file\r\n");
    fprintf(fptr,"# Last Change: %04d.%02d.%02d %02d:%02d\r\n",localTime->tm_year+1900,
        localTime->tm_mon+1,
        localTime->tm_mday,
        localTime->tm_hour,
        localTime->tm_min);
    fprintf(fptr,"my_ip=%d.%d.%d.%d\r\n", configData[0].ip[0], configData[0].ip[1], configData[0].ip[2], configData[0].ip[3]);
    fprintf(fptr,"netmask=%d.%d.%d.%d\r\n", configData[0].netmask[0], configData[0].netmask[1], configData[0].netmask[2], configData[0].netmask[3]);
    if (configData[0].options & HAS_IP_GATEWAY)
    {
        fprintf(fptr,"gateway=%d.%d.%d.%d\r\n", configData[0].gateway[0], configData[0].gateway[1], configData[0].gateway[2], configData[0].gateway[3]);
    }
    
    if (numConfig == 2)
    {
        fprintf(fptr,"my_ip_1=%d.%d.%d.%d\r\n", configData[1].ip[0], configData[1].ip[1], configData[1].ip[2], configData[1].ip[3]);
        fprintf(fptr,"netmask_1=%d.%d.%d.%d\r\n", configData[1].netmask[0], configData[1].netmask[1], configData[1].netmask[2], configData[1].netmask[3]);
        if (configData[1].options & HAS_IP_GATEWAY)
        {
            fprintf(fptr,"gateway_1=%d.%d.%d.%d\r\n", configData[1].gateway[0], configData[1].gateway[1], configData[1].gateway[2], configData[1].gateway[3]);
        }
    }
    
    fd = fileno(fptr);
    fchmod (fd, 0644);
    fchmod (fd, 0666);
    fsync(fd);
    
    fclose(fptr);
    sync();
    return NETHELPER_EXIT_SUCCESS;
}


int networkDataTo_in_addr(int *ip, struct in_addr *inAddr)
{
    unsigned char *ipArray = (unsigned char*)inAddr;
    
    // make sure all the fields are OK
    if ((ip[0] < 0) || (ip[0] > 255) ||
        (ip[1] < 0) || (ip[1] > 255) ||
        (ip[2] < 0) || (ip[2] > 255) ||
        (ip[3] < 0) || (ip[3] > 255))
    {
        return NETHELPER_EXIT_FAILURE;
    }
    
    ipArray[0] = ip[0];
    ipArray[1] = ip[1];
    ipArray[2] = ip[2];
    ipArray[3] = ip[3];
    
    return NETHELPER_EXIT_SUCCESS;
}


int setSystemNetworkData(char *ifname, int *ip, int *netmask, int *gateway)
{
    struct in_addr ipAddr;
    struct in_addr netmaskAddr;
    struct in_addr gatewayAddr;
    int result1, result2, result3, result4 = NETHELPER_EXIT_SUCCESS;
    
    if (networkDataTo_in_addr(ip, &ipAddr) != NETHELPER_EXIT_SUCCESS)
    {
        return 0x30;
    }
    if (networkDataTo_in_addr(netmask, &netmaskAddr) != NETHELPER_EXIT_SUCCESS)
    {
        return 0x30;
    }
    if (networkDataTo_in_addr(gateway, &gatewayAddr) != NETHELPER_EXIT_SUCCESS)
    {
        return 0x30;
    }
    
    // First the default gateway is removed.
    // Many default gateways are allowed in linux so this function
    // is executed until it returns error
    while(removeDefaultGateway(ifname) == NETHELPER_EXIT_SUCCESS);
    
    // Now all the settings are applied
    result1 = setIPAddress(ifname, &ipAddr);
    result2 = setNetworkMask(ifname, &netmaskAddr);
    result3 = addDefaultGateway(ifname, &gatewayAddr);
    
    // Add the broadcast route. This is very important for the Discovery service
    if(strcmp(ifname, ENET_IF0_NAME) == 0)  //12/03/2008 RAR - only for eth0 interface
    {
        result4 = addBroadcastRoute(ifname);
    }
    
    if ((result1 != NETHELPER_EXIT_SUCCESS) ||
        (result2 != NETHELPER_EXIT_SUCCESS) ||
        (result3 != NETHELPER_EXIT_SUCCESS) ||
        (result4 != NETHELPER_EXIT_SUCCESS))
    {
        return 0x31;
    }
    return NETHELPER_EXIT_SUCCESS;
}

static int convertIpConfig(IpConfig * config, IpConfigData * configData)
{
    struct in_addr inAddr;
    unsigned char *ipArray = (unsigned char*)&inAddr;
    
    if (config->ip == NULL || config->netmask == NULL)
    {
        return -1;
    }
    
    if (config->ip && inet_aton(config->ip, &inAddr) != 0)
    {
        configData->ip[0] = ipArray[0];
        configData->ip[1] = ipArray[1];
        configData->ip[2] = ipArray[2];
        configData->ip[3] = ipArray[3];
    }
    
    if (config->netmask && inet_aton(config->netmask, &inAddr) != 0)
    {
        configData->options |= HAS_IP_NETMASK;
        configData->netmask[0] = ipArray[0];
        configData->netmask[1] = ipArray[1];
        configData->netmask[2] = ipArray[2];
        configData->netmask[3] = ipArray[3];
    }
    
    if (config->gateway && inet_aton(config->gateway, &inAddr) != 0)
    {
        configData->options |= HAS_IP_GATEWAY;
        configData->gateway[0] = ipArray[0];
        configData->gateway[1] = ipArray[1];
        configData->gateway[2] = ipArray[2];
        configData->gateway[3] = ipArray[3];
    }
    
    return 0;
}


int saveWattcpNetworkData(IpConfig * config, int numConfig)
{
    IpConfigData configData[2];
    
    if (convertIpConfig(&config[0], &configData[0]) < 0)
    {
        return -1;
    }
    
    if (numConfig == 2 && convertIpConfig(&config[1], &configData[1]) < 0)
    {
        return -2;
    }
    
    if (setWattcpNetworkData(configData, numConfig) != NETHELPER_EXIT_SUCCESS)
    {
        return -3;
    };
    
    return 0;
}

int saveWattcpNetworkDataEx(IpConfigEx * config, int numConfig)
{
    int i;
    FILE *fptr;
    int fd;
    ConfigParser cp;

    ConfigParser_Init(&cp, '=');

    if (!ConfigParser_Load(&cp, WATTCP_CFG_FILENAME))
    {
        return 0x1B;
    }

    // Get current time
    time_t currentTime;
    struct tm *localTime;
    time(&currentTime);
    localTime = localtime(&currentTime);
    
    if ((fptr = fopen(WATTCP_CFG_FILENAME, "w")) == NULL)
    {
        ConfigParser_Cleanup(&cp);
        return 0x1B;
    }
    
    fprintf(fptr,"# ProSoft Technology Inc.\r\n");
    fprintf(fptr,"# Automatically generated file\r\n");
    fprintf(fptr,"# Last Change: %04d.%02d.%02d %02d:%02d\r\n",localTime->tm_year+1900,
        localTime->tm_mon+1,
        localTime->tm_mday,
        localTime->tm_hour,
        localTime->tm_min);

    for (i = 0; i < numConfig; i++)
    {
        switch (config[i].type)
        {
        case IP_CFG_PORT1:
            if (!ConfigParser_ChangeItem(&cp, "my_ip", config[i].ip))
            {
                ConfigParser_AddItemString(&cp, "my_ip", config[i].ip);
            }

            if (!ConfigParser_ChangeItem(&cp, "netmask", config[i].netmask))
            {
                ConfigParser_AddItemString(&cp, "netmask", config[i].netmask);
            }

            if (!ConfigParser_ChangeItem(&cp, "gateway", config[i].gateway))
            {
                ConfigParser_AddItemString(&cp, "gateway", config[i].gateway);
            }

        	break;

        case IP_CFG_PORT2:

            if (!ConfigParser_ChangeItem(&cp, "my_ip_1", config[i].ip))
            {
                ConfigParser_AddItemString(&cp, "my_ip_1", config[i].ip);
            }

            if (!ConfigParser_ChangeItem(&cp, "netmask_1", config[i].netmask))
            {
                ConfigParser_AddItemString(&cp, "netmask_1", config[i].netmask);
            }

            if (!ConfigParser_ChangeItem(&cp, "gateway_1", config[i].gateway))
            {
                ConfigParser_AddItemString(&cp, "gateway_1", config[i].gateway);
            }

            break;

        case IP_CFG_WLAN:

            if (!ConfigParser_ChangeItem(&cp, "Radio_IP", config[i].ip))
            {
                ConfigParser_AddItemString(&cp, "Radio_IP", config[i].ip);
            }

            if (!ConfigParser_ChangeItem(&cp, "Radio_Subnet_Mask", config[i].netmask))
            {
                ConfigParser_AddItemString(&cp, "Radio_Subnet_Mask", config[i].netmask);
            }

            if (!ConfigParser_ChangeItem(&cp, "Radio_Default_Gateway", config[i].gateway))
            {
                ConfigParser_AddItemString(&cp, "Radio_Default_Gateway", config[i].gateway);
            }

            break;
        }
    }

    ConfigParser_Save2(&cp, fptr);
    ConfigParser_Cleanup(&cp);

    fd = fileno(fptr);
    fchmod (fd, 0644);
    fchmod (fd, 0666);
    fsync(fd);
    
    fclose(fptr);
    sync();

    return NETHELPER_EXIT_SUCCESS;
}
