"""
Run with 
    > cd /path/to/extracted/zip
    > python -m PlxConnect.CLI -h
    
This will show a help message and help you get started collecting data.
For additional examples, see the included README.txt in this folder.
    
Command line interface for interacting with the PLX51DL products via REST API.
For more information on this API, see the PLX51 User Manual on the Prosoft website.
https://www.prosoft-technology.com/Products/Gateways/PLX5x/Data-Logger-Plus
"""
import argparse
import PlxConnect


parser = argparse.ArgumentParser(
    formatter_class=argparse.RawTextHelpFormatter,
    allow_abbrev=False,
    description = "A demo CLI for interacting with the PLX51 Data Logger's REST API"
)
subparsers = parser.add_subparsers(help="sub-commands", dest="command")

# IP is the only required global field
parser.add_argument(
    '--ip',
    type=str,
    required=True,
    help="The IP address of the module\n" \
        "i.e. 192.168.10.75"
)

# reqGenSts
gen_status_parser = subparsers.add_parser("GeneralStatus", help="Get device status with reqGenSts")
# reqCacheStats
cache_stats_parser = subparsers.add_parser("CacheStats", help="Get device cache statistics with reqCacheStats")

# reqGetTrendData and reqGetTrendDataUTC
trend_data_parser = subparsers.add_parser(
    "GetTrendData",
    help="Gather logged data over a given range of time with Gregorian dates. UTC timezone."
)

trend_data_parser.add_argument(
    '--tags',
    nargs=1,
    required=False,
    default=["all"],
    action="store",
    type=str,
    help="(default: all) List of 1 to 5 comma-separated tag indexes to retrieve.  To read all tags, pass 'all' instead." \
    "Tags must be valid and defined in the PLX51 config.  This can be verified with the GetTagNames subcommand." \
    "Example: python -m CLI GetTrendData --duration \"Last 5 min\" --tags 1,22,44"
)
    
trend_data_parser.add_argument(
    '--duration',
    nargs=1,
    required=False,
    default=None,
    action="store",
    choices=PlxConnect.TREND_DATA_DURATIONS,
    type=str,
    help="Duration to search for trend data.  Use 'Custom Dates' to specify start/stop via cmdline arguments."
)

trend_data_parser.add_argument(
    '--start',
    nargs=2,
    required=False,
    action="store",
    type=str,
    help="\"YY/MM/DD HH:MM:SS\" - " \
    "Start DateTime if using 'Custom Dates' as a duration for getting TrendData."
)

trend_data_parser.add_argument(
    '--stop',
    nargs=2,
    required=False,
    action="store",
    type=str,
    help="\"YY/MM/DD HH:MM:SS\" - " \
    "Stop DateTime if using 'Custom Dates' as a duration for getting TrendData."
)


# TrendDataUTC Parser
utc_trend_data_parser = subparsers.add_parser(
    "GetTrendDataUTC",
    help="Gather logged data over a given range of time using epoch timestamps. UTC Timezone."
)

utc_trend_data_parser.add_argument(
    '--tags',
    nargs=1,
    required=False,
    default=["all"],
    action="store",
    type=str,
    help="List of 1 to 5 comma-separated tag indexes to retrieve.  To read all tags (default), pass 'all' instead." \
    "Tags must be valid and defined in the PLX51 config.  This can be verified with the GetTagNames subcommand." \
    "Example: python -m CLI GetTrendData --duration \"Last 5 min\" --tags 1,22,44"
)
    
utc_trend_data_parser.add_argument(
    '--duration',
    nargs=1,
    required=False,
    default=None,
    action="store",
    choices=PlxConnect.TREND_DATA_DURATIONS,
    type=str,
    help="Duration to search for trend data.  Use 'Custom Dates' to specify start/stop via cmdline arguments."
)

utc_trend_data_parser.add_argument(
    '--start',
    nargs=1,
    required=False,
    action="store",
    type=str,
    help="Start time in seconds since epoch if using 'Custom Dates' as a duration for getting TrendData."
)

utc_trend_data_parser.add_argument(
    '--stop',
    nargs=1,
    required=False,
    action="store",
    type=str,
    help="Stop time in seconds since epoch if using 'Custom Dates' as a duration for getting TrendData."
)

# reqGetCacheRecords
get_cache_parser = subparsers.add_parser("GetCacheRecords", help="Retrieve cache records with reqGetCacheRecords")
get_cache_parser.add_argument(
    'index',
    action="store",
    type=int,
    help="Starting index for reading cache records (0-16,777,216)"
)

get_cache_parser.add_argument(
    'count',
    action="store",
    type=int,
    help="Number of cache records to read in this request (1-5)"
)

# reqGetTagNames
get_tags_parser = subparsers.add_parser("GetTagNames", help="Get a listing of tags with reqGetTagNames")
get_tags_parser.add_argument(
    'index',
    action="store",
    type=int,
    help="Starting index for reading tag names (0-200)"
)

get_tags_parser.add_argument(
    'count',
    action="store",
    type=int,
    help="Number of tag names to read in this request (1-4)"
)

# reqUnloadLogIdx
unload_idx_parser = subparsers.add_parser("UnloadIdxUpdate", help="Set upload index on the target PLX51")
unload_idx_parser.add_argument(
    'index',
    action="store",
    type=int,
    help="New value for UnloadIdx on target device"
)

reset_logidx_parser = subparsers.add_parser("LogIndexReset", help="!!! WARNING: This will erase all unsaved cache data !!! Reset the log index on the PLX51 to 0.")

# Generic arguments for printing and logging
parser.add_argument(
    '--print',
    required=False,
    default=True,
    action=argparse.BooleanOptionalAction,
    help="Enable/disable printing of REST API message data to terminal."
)

parser.add_argument(
    '--logging',
    required=False,
    default=None,
    choices=["requests", "responses", "all"],
    action="store",
    type=str,
    help="Choose which messages to (requests) or from (responses) the REST API to save in the session log."
)

parser.add_argument(
    '--log_dir',
    required=False,
    action="store",
    type=str,
    help="Directory to save log files i.e. C:/Sessions/ (defaults to script directory)"
)

if __name__ == '__main__':
    args = parser.parse_args()
    plx_connection = PlxConnect.Plx5xConnection(
        args.ip,
        print_en = args.print,
        log_requests = args.logging == "all" or args.logging == "requests",
        log_responses = args.logging == "all" or args.logging == "responses",
        log_dir = args.log_dir
    )
    
    if args.command == "GetTrendData":
        # Prepare tags cmdline args for fncall
        if args.tags[0].lower() == "all":
            tags = []
        else:
            tags = [int(i) for i in args.tags[0].split(",")]
        
        # Send different requests based on duration.  At this point, duration content
        # has been verified by argparsing, so passing directly is safe.
        if args.duration is None or args.duration[0] == "Custom Dates":
            # Send start/stop if custom dates are required.
            plx_connection.get_trend_data(
                duration_str = None,
                tag_idxs = tags,
                start_time = " ".join(args.start),
                stop_time = " ".join(args.stop)
            )
        else:
            # Otherwise don't send start/stop args (they'll be ignored if they are, regardless)
            plx_connection.get_trend_data(
                duration_str = args.duration[0],
                tag_idxs = tags
            )
            
    elif args.command == "GetTrendDataUTC":
        # Prepare tags cmdline args for fncall
        if args.tags[0].lower() == "all":
            tags = []
        else:
            tags = [int(i) for i in args.tags[0].split(",")]

        # Send different requests based on duration.  At this point, duration content
        # has been verified by argparsing, so passing directly is safe.
        if args.duration is None or args.duration[0] == "Custom Dates":
            # Send start/stop if custom dates are required.
            plx_connection.get_trend_data_utc(
                duration_str = None,
                tag_idxs = tags,
                start_time = args.start[0],
                stop_time = args.stop[0],
            )
        else:
            # Otherwise don't send start/stop args (they'll be ignored if they are, regardless)
            plx_connection.get_trend_data_utc(
                duration_str = args.duration[0],
                tag_idxs = tags
            )
            
    elif args.command == "GetCacheRecords":
        if args.count > 5:
            raise ValueError("GetCacheRecords read count cannot be > 5")
            
        plx_connection.get_cache_records(
            args.index,
            args.count
        )
        
    elif args.command == "GeneralStatus":
        plx_connection.get_general_status()
        
    elif args.command == "CacheStats": 
        plx_connection.get_cache_statistics()
        
    elif args.command == "GetTagNames":
        plx_connection.get_tag_names(
            args.index,
            args.count 
        )

    elif args.command == "UnloadIdxUpdate":
        plx_connection.unload_log_index_update(args.index)
        
    elif args.command == "LogIndexReset":
        plx_connection.log_index_reset()

    else:
        print("No Command chosen.  Run with -h for options")
    