"""
This script is an example of how to utilize the Plx5xConnection class methods in a script.
Python method names match the wording used in the PLX51 user manual REST API descriptions.
For more information about the role and function of these methods, review the PLX51 UM and 
search for these keywords.

Note the below examples contains minimal input checking and error handling.
"""

import PlxConnect

# NOTE: Change this IP address to match your setup
module_ip = "192.168.0.250"

# Init the Connection to the PLX51.  Note all values are manually set here.
# Check Plx5xConnection class definition for a list of default values.
rest_client = PlxConnect.Plx5xConnection(
    module_ip,
    print_en=True,
    log_requests=False,     # Start with all logging disabled
    log_responses=False,
    log_dir=None
)

### Getting and reporting General Statuses
gen_stat = rest_client.get_general_status()

# Show config CRC, for example
print("Current config CRC is {}".format(gen_stat["ConfigCRC"]))
# Note that responses are in the form of Python dictionaries, so returned
# keys and values can be accessed in the usual way.  i.e.
print("Keys from GeneralStatus call were {}".format(gen_stat.keys()))

### Getting tag indexes from the device (how to get from names to indexes)
tags_list = rest_client.get_tag_names(0, 4)
print(tags_list)

# Disable response printing 
rest_client.print_en = False

# Get the ConfigCRC from general status before collecting tags.  If the CRC changes, we can
# assume the tags may have changed and force a re-collection later on.
start_crc = rest_client.get_general_status()["ConfigCRC"]

# Get the first 4 tags (note these tags must be present in the PLX51 configuration)
test_tags = rest_client.get_tag_names(0, 4)

# Ensure that our CRC hasn't changed!
assert start_crc == rest_client.get_general_status()["ConfigCRC"]

# Start printing again
rest_client.print_en = True

# Get Cache statistics
rest_client.get_cache_statistics()

# Get records from the cache
index = 1
count = 5
rest_client.get_cache_records(index, count)

### NOTE: The below should be uncommented to use the UnloadLogIndexUpdate API call
# Unload the log index 
#log_index = 5
#rest_client.unload_log_index_update(log_index)


### NOTE: The below should be uncommented to use the LogIndexReset API call
# Reset the log index
#rest_client.log_index_reset()


### Trend Data Examples.  For more information about this method, see the PlxConnect.py
# Get trend data with preset duration, tags 1-4, standard times.
duration = "Last 5 min"
trend_tags = [1, 2, 3, 4]
rest_client.get_trend_data(duration_str=duration, tag_idxs=trend_tags)

# Get trend data for all tags with custom duration, standard times
start_sample = "2023/01/17 21:20:00"
stop_sample = "2023/01/17 22:26:00"
rest_client.get_trend_data(start_time=start_sample, stop_time=stop_sample)

# Get trend data for all tags with custom duration, UTC times
# Note that both get_trend_data() and get_trend_data_utc() take times in the UTC timezone, but
# get_trend_data_utc() accepts epoch/ctime instead of gregorian dates.
start_sample_s = "1673985000"
stop_sample_s = "1673985060"
rest_client.get_trend_data_utc(start_time=start_sample_s, stop_time=stop_sample_s)
