/* tw-ldm-interface.c
 This file contains the 'main' function. Program execution begins and ends there.

This file contains sample code for the MVI56E-LDM ThingWorx add-on.
(c) 2019, ProSoft Technology, All Rights Reserved
*/

#include <twApi.h>
#include <twExt.h>
#include <twMacros.h>
#include <stdio.h>
#include <string.h>
#include "tw-ldm-interface-lib.h"
#include <../messaging/twMessaging.h>

#ifndef WIN32
#include <syslog.h>
#include <unistd.h>
#endif

uint32_t get_tick_count(void) {

#ifdef WIN32
	return GetTickCount();
#else
#include <unistd.h>
#include <sys/time.h>
	struct timeval ts;
	gettimeofday(&ts, 0);
	return ts.tv_sec * 1000 + (ts.tv_usec / 1000);
#endif
}
volatile char shutting_down = 0;

void tw_ldm_log(enum tw_ldm_log_level log_level, const char *format, ...)
{
	va_list arguments;
	va_start(arguments, format);
#ifdef WIN32	
	vprintf(format, arguments);
	printf("\n");
#else
	vsyslog(log_level, format, arguments);
#endif
	va_end(arguments);
}

void open_logging()
{
#ifndef WIN32
	// setlogmask(LOG_UPTO(LOG_NOTICE));
	openlog("TW-LDM", LOG_CONS | LOG_NDELAY | LOG_PERROR, LOG_USER);
#endif
}

void close_logging()
{
#ifndef WIN32
closelog();
#endif
}

int tw_ldm_connect(void)
{
	int result = TW_OK;
	shutting_down = FALSE;
	tw_ldm_log(TW_LDM_LOG_NOTICE, "Connecting to ThingWorx server at [%s:%d], retry count = %d.", gl_app_config.tw_server.hostname, gl_app_config.tw_server.port, gl_app_config.tw_server.connection_retry_count);
	
	if (gl_app_config.tw_server.use_https && !gl_app_config.tw_server.disable_certificate_validation)
	{
		char* path_to_file;
		if (gl_app_config.tw_server.path_to_root_ca_certificate_file)
			path_to_file = gl_app_config.tw_server.path_to_root_ca_certificate_file;
		else
			path_to_file = "root_ca.cer";

		result = twApi_LoadCACert(path_to_file, 0);
		if (result != TW_OK)
			tw_ldm_log(TW_LDM_LOG_ERROR, "Failed to load CA certificate from file [%s], result = [%d]", path_to_file, result);
	}

	if (result == TW_OK)
	{
		char is_connected;
		result = twApi_Connect(gl_app_config.tw_server.timeout, 1); // gl_app_config.tw_server.connection_retry_count);
		is_connected = twApi_isConnected();
		if (TW_OK == result && is_connected )
			tw_ldm_log(TW_LDM_LOG_NOTICE, "Successfully connected to ThingWorx server");
		else
		{
			if (result == TW_OK)
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "twApi_Connect returned with result = %d, but twApi_isConnected returned %d", result, (int) is_connected);
				result = 1;
			}
			else 
				tw_ldm_log(TW_LDM_LOG_ERROR, "twApi_Connect failed with result = %d", result);
		}
	}

	return result;
}

int tw_ldm_disconnect(void)
{
	// Set PLC connection state in each thing to false:
	tw_thing* thing;
	int number_of_updates = 0;
	DATETIME current_time = twGetSystemTime(TRUE);

	thing = gl_app_config.tw_server.things;
	while (thing)
	{
		if (thing->is_connected_property_name && strlen(thing->is_connected_property_name) > 0)
		{
			twApi_SetSubscribedPropertyVTQ(thing->name, thing->is_connected_property_name, TW_MAKE_BOOL(tw_ldm_is_connected_to_plc()), 
				current_time, TW_QUALITY_UNKNOWN, 0, 0);
			number_of_updates++;
		}
		thing = thing->next;
	}
	if (number_of_updates)
	{
		int wait_time = gl_app_config.tw_server.polling_rate * 1000 * 2;
		if (wait_time < 3000000)
			wait_time = 3000000;
		else if (wait_time > 30000000)
			wait_time = 30000000;

		tw_ldm_log(TW_LDM_LOG_NOTICE, "Synchronizing state of things with ThingWorx server...");
		TW_PUSH_PROPERTIES_FOR(NULL, TW_PUSH_CONNECT_LATER);
		usleep(wait_time);
		tw_ldm_log(TW_LDM_LOG_NOTICE, "Synchronizing state of things with ThingWorx server is complete.");
	}

	int result = twApi_Disconnect("Normal disconnection");

	return result;
}

int tw_ldm_start(void)
{
	// Threads are expensive, so create only one thread:
	const uint32_t number_of_threads = 1;
	twExt_Start(gl_app_config.tw_server.polling_rate, TW_THREADING_TASKER, number_of_threads);
	return 0;
}

int tw_ldm_stop(void)
{
	int result = twExt_Stop();
	if (result != TW_OK)
		tw_ldm_log(TW_LDM_LOG_ERROR, "ThingWorx SDK stopped with result %d", result);

	return result;
}

extern int tw_ldm_clean(void)
{
	int result;
	tw_ldm_free_config(&gl_app_config);
	result = tw_ldm_disconnect();
	result = tw_ldm_stop();

	result = twApi_Delete();
	tw_ldm_log(TW_LDM_LOG_NOTICE, "********** ThingWorx API closed with code [%d] ******************", result);
	close_logging();
	return result;
}


int tw_ldm_is_connected(void)
{
	return twApi_isConnected();
}


int tw_ldm_is_connecting(void)
{
	return twApi_ConnectionInProgress();
}

/* Global variable, application configuration */
tw_ldm_config gl_app_config;

get_app_key_callback_function_type get_app_key_callback_function = NULL;

void app_key_callback(char* appKeyBuffer, unsigned int maxLength)
{
	static char* default_app_key = "8918cf96-6de4-400b-801f-ac6cdb450279";
  size_t len;
	memset(appKeyBuffer, 0, maxLength);

	/* If callback function is defined, then call it to get the key: */
	if (get_app_key_callback_function)
	{
		get_app_key_callback_function(appKeyBuffer, maxLength);
		if (appKeyBuffer[0])
			return;
	}

	/* If callback function is not defined, or it did not set the key, then use the one which was read from configuration file: */
	if (gl_app_config.tw_server.app_key[0])
	{
  	len = strlen(gl_app_config.tw_server.app_key);
  	if (len > 0 && maxLength > len)
    	strncpy(appKeyBuffer, gl_app_config.tw_server.app_key, len);
    if (appKeyBuffer[0])
     	return;
	}

	/* As last resort, use hard coded default key (which will not work in production though, but for initial demo should be OK) */
  len = strlen(default_app_key);
  if (len > 0 && maxLength > len)
	  strncpy(appKeyBuffer, default_app_key, len);
}

int createThings();

void custom_log_function(enum LogLevel level, const char * timestamp, const char * message)
{
	enum tw_ldm_log_level syslog_level;
	switch (level)
	{
	case TW_TRACE:
	case TW_DEBUG:
		syslog_level = TW_LDM_LOG_DEBUG;
		break;
	case TW_INFO:
	case TW_AUDIT:
		syslog_level = TW_LDM_LOG_INFO;
		break;
	case TW_WARN:
		syslog_level = TW_LDM_LOG_WARNING;
		break;
	case TW_ERROR:
	case TW_FORCE:
		syslog_level = TW_LDM_LOG_ERROR;
		break;
	default:
		syslog_level = TW_LDM_LOG_ERROR;
		break;
	}
	tw_ldm_log(syslog_level, message);
}

int tw_ldm_initialize(const char* path_to_conf_file, get_app_key_callback_function_type app_key_cb_function)
{
	int result;
	const char* path_to_config_file = "config.json";

	result = twLogger_SetFunction(custom_log_function);

	open_logging();

	memset(&gl_app_config, 0, sizeof(gl_app_config));
	if (path_to_conf_file)
		path_to_config_file = path_to_conf_file;
	
	get_app_key_callback_function = app_key_cb_function;

	/* Configure ThingWorx Logging */
	twLogger_SetLevel(TW_ERROR);
	twLogger_SetIsVerbose(FALSE); 
	
	/* Load configuration */
	result = load_and_parse_configuration_file(path_to_config_file, &gl_app_config);
	if (result != 0)
		return result;
	
	tw_ldm_log(TW_LDM_LOG_NOTICE, "Initializing ThingWorx API");

	result = twApi_Initialize(gl_app_config.tw_server.hostname, gl_app_config.tw_server.port, TW_URI, app_key_callback, NULL, MESSAGE_CHUNK_SIZE, MESSAGE_CHUNK_SIZE, TRUE);
	if (TW_OK != result) 
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "ThingWorx API initialization failed: result = [%d]", result);
		return result;
	}
	else
		tw_ldm_log(TW_LDM_LOG_NOTICE, "ThingWorx API initialization succeeded");

	/* Allow self signed or unvalidated certs (Neither should be used in production) */
	/* twApi_SetSelfSignedOk(); */ /* Requires a certificate that is self signed */

	if (gl_app_config.tw_server.disable_certificate_validation)
		twApi_DisableCertValidation(); 

	if (!gl_app_config.tw_server.use_https) 
	{
		twApi_DisableEncryption();
	}

	result = createThings();

	if (result != 0)
	{
		tw_ldm_log(TW_LDM_LOG_ERROR, "createThings failed with result = %d", result);
		return result;
	}

	// This call allocates big chunk of memory
	// Total size of allocations depend on thread model and number of threads, somewhere from 20 Mb to 60 Mb. 
	// As a result, total memory used by the process can become higher than 100%: 137% was observed.
	// After changing model from MULTY to TASKER, with thread count 1 it seems that memory usage become acceptable, about 45% when connected over https.
	if (result == 0)
		result = tw_ldm_start();

	return 0;
}

void poll_thing(char *thing_name)
{
	int result;

	if (!thing_name)
		return;
	{
		uint32_t current_tick = get_tick_count();
		tw_thing* current_thing = gl_app_config.tw_server.things;
		tw_tag* tag = gl_app_config.tags;
		int number_of_updated_tags = 0;
		twPrimitive* tw_value;
		DATETIME current_time = twGetSystemTime(TRUE);

		while (current_thing)
		{
			if (strcmp(current_thing->name, thing_name) == 0)
				break;
			current_thing = current_thing->next;
		}

		if (!current_thing)
		{
			// TODO: Log
			return;
		}
		
		// Set property indicating if the PLC is connected and running:
		if (current_thing->is_connected_property_name && strlen(current_thing->is_connected_property_name) > 0)
		{
			tw_value = TW_MAKE_BOOL(tw_ldm_is_connected_to_plc());
			twApi_SetSubscribedPropertyVTQ(thing_name, current_thing->is_connected_property_name, tw_value, current_time, 
				(shutting_down ? TW_QUALITY_UNKNOWN : TW_QUALITY_GOOD), 0, 0);
			number_of_updated_tags++;
		}

		// Loop over all tags and process ones which are mapped to this thing:
		while (tag)
		{
			if (strcmp(tag->thing, thing_name) == 0)
			{
				tw_thing_property* property = current_thing->properties;
				while (property)
				{
					if (strcmp(property->name, tag->property_name) == 0)
					{
						char* quality;
						if ((current_tick - tag->poll_time) < tag->scan_rate)
							break;
						result = tw_ldm_read_value(tag, &tag->value);
						switch (result)
						{
						case 0:
							quality = TW_QUALITY_GOOD;
							break;
						case TW_LDM_DEVICE_NOT_CONNECTED:
							quality = TW_QUALITY_UNKNOWN;
							break;
						default:
							quality = TW_QUALITY_BAD;
							break;
						}

						tag->poll_time = current_tick;
						// if (result == 0)
						{
							number_of_updated_tags++;
							switch (property->type)
							{
							case TW_BOOLEAN:
								tw_value = TW_MAKE_BOOL(tag->value.value.int8_value);
								twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
								break;
							case TW_NUMBER:
								switch (tag->data_type)
								{
								case plc_type_LREAL:
									tw_value = TW_MAKE_NUMBER(tag->value.value.real64_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								case plc_type_REAL:
									tw_value = TW_MAKE_NUMBER(tag->value.value.real32_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								case plc_type_SINT:
								case plc_type_USINT:
									tw_value = TW_MAKE_NUMBER(tag->value.value.int8_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								case plc_type_INT:
								case plc_type_UINT:
									tw_value = TW_MAKE_NUMBER(tag->value.value.int16_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								case plc_type_DINT:
								case plc_type_UDINT:
									tw_value = TW_MAKE_NUMBER(tag->value.value.int32_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								}
								break;
							case TW_INTEGER:
								switch (tag->data_type)
								{
								case plc_type_LREAL:
									tw_value = TW_MAKE_NUMBER(tag->value.value.real64_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								case plc_type_REAL:
									tw_value = TW_MAKE_INT((int)tag->value.value.real32_value);
									// TW_SET_PROPERTY(thing_name, property->name, tw_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, tag->property_name, tw_value, current_time, quality, 0, 0);
									break;
								case plc_type_SINT:
								case plc_type_USINT:
									tw_value = TW_MAKE_INT(tag->value.value.int8_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								case plc_type_INT:
								case plc_type_UINT:
									tw_value = TW_MAKE_INT(tag->value.value.int16_value);
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								case plc_type_DINT:
								case plc_type_UDINT:
									tw_value = TW_MAKE_INT(tag->value.value.int32_value);
									// TW_SET_PROPERTY
									twApi_SetSubscribedPropertyVTQ(thing_name, property->name, tw_value, current_time, quality, 0, 0);
									break;
								}
								break;
							case TW_STRING:
								tw_value = TW_MAKE_STRING(&tag->value.value.string_value[4]);
								// TW_SET_PROPERTY(thing_name, property->name, tw_value);
								twApi_SetSubscribedPropertyVTQ(thing_name, tag->property_name, tw_value, current_time, quality, 0, 0);
								break;
							default:
								// TODO: log
								break;
							}
						}
						// else
						{
							if (result != 0 && result != TW_LDM_DEVICE_NOT_CONNECTED)
								tw_ldm_log(TW_LDM_LOG_INFO, "Tag [%s]: failed to read value, result = %d", tag->tag_name, result);
						}
						// Call to twApi_SetSubscribedPropertyVTQ transfers ownership of "value", so no need to free it here:
						//if (tw_value)
						//{
						//	if (tw_value->type == TW_STRING && tw_value->val.bytes.data)
						//		TW_FREE(tw_value->val.bytes.data);
						//	TW_FREE(tw_value);
						//}
						break;
					}
					property = property->next;
				}
				if (!property)
				{
					// TODO: Log
				}
			}
			tag = tag->next;
		}
		if (number_of_updated_tags > 0)
		{
			result = TW_PUSH_PROPERTIES_FOR(thing_name, TW_PUSH_CONNECT_LATER);
			if (result != TW_OK)
				tw_ldm_log(TW_LDM_LOG_WARNING, "Failed to push properties to ThingWorx, result = [%d]", result);
		}
	}
}

enum msgCodeEnum set_property_service_handler(const char * entityName, const char * serviceName, twInfoTable * params, twInfoTable ** content, void * userdata) 
{
	double value;
	char * property_name = NULL;
	int result = 0;
	tw_tag* current_tag;

	if (!params || !content || !entityName) {
		TW_LOG(TW_ERROR, "set_property_service_handler - NULL params or content pointer");
		return TWX_BAD_REQUEST;
	}

	///* Get property name from input parameter: */
	result = twInfoTable_GetString(params, "name", 0, &property_name);

	if (result == TW_OK && property_name)
	{
		// Find mapped tag:
		result = 1;
		current_tag = gl_app_config.tags;
		while (current_tag)
		{
			if (strcmp(current_tag->thing, entityName) == 0 && strcmp(current_tag->property_name, property_name) == 0)
			{
				plc_value plc_val;

				if (current_tag->access_mode != PlcTagAccessMode_ReadWrite)
				{
					tw_ldm_log(TW_LDM_LOG_WARNING, "Tag [%s]: read only, ignoring attempt to write", current_tag->tag_name);
					result = TWX_NOT_ACCEPTABLE;
					break;
				}
				memset(&plc_val, 0, sizeof(plc_val));
				plc_val.type = current_tag->data_type;

				if (current_tag->data_type == plc_type_STRING82)
				{
					char* string_value = NULL;
					result = twInfoTable_GetString(params, "value", 0, &string_value);
					if (result == TW_OK)
					{
						int32_t len = 0;
						if (string_value != NULL)
						{
							 len = strlen(string_value);
						}
						if (len > 82)
							len = 82;
						memcpy(plc_val.value.string_value, (char*) &len, sizeof(len));
						if (len > 0)
						{
							memcpy(plc_val.value.string_value + sizeof(len), string_value, len);
						}
					}
					if (string_value)
						TW_FREE(string_value);
				}
				else
				{
					result = twInfoTable_GetNumber(params, "value", 0, &value);
					if (result == TW_OK)
					{
						switch (current_tag->data_type)
						{
						case plc_type_BOOL:
							plc_val.value.int8_value = (value == 0) ? 0 : 1;
							break;
						case plc_type_SINT:
						case plc_type_USINT:
						case plc_type_BYTE:
							plc_val.value.int8_value = (char)value;
							break;
						case plc_type_INT:
						case plc_type_UINT:
						case plc_type_WORD:
							plc_val.value.int16_value = (int16_t)value;
							break;
						case plc_type_DINT:
						case plc_type_UDINT:
						case plc_type_DWORD:
							plc_val.value.int32_value = (int32_t)value;
							break;
						case plc_type_LINT:
						case plc_type_ULINT:
						case plc_type_LDWORD:
							plc_val.value.int64_value = (int64_t)value;
							break;
						case plc_type_REAL:
							plc_val.value.real32_value = (float)value;
							break;
						case plc_type_LREAL:
							plc_val.value.real64_value = (double)value;
							break;
						default:
							plc_val.type = 0;
							break;
						}
					}
				}
				if (plc_val.type != 0)
				{
					result = tw_ldm_write_value(current_tag, &plc_val);
					if (result != 0)
					{
						if (result != TW_LDM_DEVICE_NOT_CONNECTED)
							tw_ldm_log(TW_LDM_LOG_WARNING, "Tag [%s]: failed to write value, result = %d", current_tag->tag_name, result);
					}
				}
				else
				{
					tw_ldm_log(TW_LDM_LOG_ERROR, "Tag [%s]: failed to write value, internal error.", current_tag->tag_name);
					result = 2;
				}

				if (result == 0)
				{
					// TODO - we could optionally: 
					// 1. Read it back and verify the value is really written;
					// 2. Update property value to this new value.
				}
				break;
			}
			current_tag = current_tag->next;
		}
	}
	/* Return Result */
	*content = twInfoTable_CreateFromInteger("result", result);
	if (property_name)
		TW_FREE(property_name);

	if (*content)
		return TWX_SUCCESS;
	else
		return TWX_INTERNAL_SERVER_ERROR;
}

enum msgCodeEnum get_status_service_handler(const char * entityName, const char * serviceName, twInfoTable * params, twInfoTable ** content, void * userdata)
{
	int result;
	int32_t is_verbose;
	uint16_t max_size;
#define STATUS_BUFFER_SIZE 32
	static char static_buffer[STATUS_BUFFER_SIZE+1];
	char* buffer;
	if (!content) 
	{
		TW_LOG(TW_ERROR, "get_status_service_handler: invalid argument \"params\"");
		return TWX_BAD_REQUEST;
	}

	if (params)
	{
		result = twInfoTable_GetInteger(params, "verbose", 0, &is_verbose);
		if (result != TW_OK)
			return TWX_BAD_REQUEST;
	}
	else
		is_verbose = FALSE;

	// If mode is verbose, then memory for returned value is allocated from the heap, and ownership to it is transfered to the TW SDK.
	// Usually mode will not be verbose, in this case static buffer is used, which eliminates allocation/deallocation steps.
	if (is_verbose)
	{
		buffer = NULL; //Tells that memory must be allocated by the tw_ldm_get_status function.
		max_size = 0;
	}
	else
	{
		buffer = static_buffer;
		max_size = STATUS_BUFFER_SIZE;
		memset(static_buffer, 0, sizeof(static_buffer));
	}
	if (tw_ldm_get_status(is_verbose, &buffer, max_size) == 0)
	{
		*content = twInfoTable_CreateFromString("result", buffer, is_verbose ? FALSE : TRUE); 

		if (*content)
			return TWX_SUCCESS;
	}
		
	return TWX_INTERNAL_SERVER_ERROR;
}

int createThings()
{
	//char logFileNameBuffer[255];
	//char* fullPathToLogs;
	#define MAX_NUMBER_OF_TEMPLATES 64
	char* template_names[MAX_NUMBER_OF_TEMPLATES];
	memset(template_names, 0, sizeof(template_names));
	int template_index = 0;
	tw_thing* current_thing = gl_app_config.tw_server.things;

	while (current_thing)
	{
		int index;
		int scan_function_registered = 0;
		
		/* Declare Properties */
		tw_thing_property* current_property = current_thing->properties;
		TW_MAKE_THING(current_thing->name, current_thing->template_name);
		
		for (index = 0; index < template_index; index++)
		{
			if (strcmp(template_names[index], current_thing->template_name) == 0)
			{
				scan_function_registered = 1;
				break;
			}
		}
		if (!scan_function_registered)
		{
			if (template_index < MAX_NUMBER_OF_TEMPLATES)
			{
				template_names[template_index] = current_thing->template_name;
				twExt_RegisterPolledTemplateFunction(poll_thing, current_thing->template_name);
				template_index++;
			}
			else
			{
				tw_ldm_log(TW_LDM_LOG_ERROR, "Too many [%d] thing templates are used", template_index + 1);
				return -1;
			}
		}

		if (current_thing->is_connected_property_name && strlen(current_thing->is_connected_property_name) > 0)
		{
			TW_PROPERTY(current_thing->is_connected_property_name, "", TW_BOOLEAN);
		}

		while (current_property)
		{
			tw_property_aspect* current_aspect = current_property->aspects;
			TW_PROPERTY(current_property->name, current_property->description, current_property->type);
			while (current_aspect)
			{
				/* Declare aspects: */
				switch (current_aspect->type)
				{
				case TW_BOOLEAN:
					TW_ADD_BOOLEAN_ASPECT(current_property->name, current_aspect->name, current_aspect->value.integerValue);
					break;
				case TW_NUMBER:
					TW_ADD_NUMBER_ASPECT(current_property->name, current_aspect->name, current_aspect->value.floatValue);
					break;
				}
				current_aspect = current_aspect->next;
			}
			current_property = current_property->next;
		}

		TW_SERVICE("SetNumericProperty",
			"Set property value of numeric data type",
			TW_MAKE_DATASHAPE(TW_SHAPE_NAME_NONE,
				TW_DS_ENTRY("name", TW_NO_DESCRIPTION, TW_STRING),
				TW_DS_ENTRY("value", TW_NO_DESCRIPTION, TW_NUMBER)
			),
			TW_NUMBER,
			TW_NO_RETURN_DATASHAPE,
			set_property_service_handler);

		TW_SERVICE("SetStringProperty",
			"Set property value of string data type",
			TW_MAKE_DATASHAPE(TW_SHAPE_NAME_NONE,
				TW_DS_ENTRY("name", TW_NO_DESCRIPTION, TW_STRING),
				TW_DS_ENTRY("value", TW_NO_DESCRIPTION, TW_NUMBER)
			),
			TW_NUMBER,
			TW_NO_RETURN_DATASHAPE,
			set_property_service_handler);

		TW_SERVICE("GetStatus",
			"Get Status",
			TW_MAKE_DATASHAPE(TW_SHAPE_NAME_NONE,
				TW_DS_ENTRY("verbose", TW_NO_DESCRIPTION, TW_INTEGER)
			),
			TW_STRING,
			TW_NO_RETURN_DATASHAPE,
			get_status_service_handler);

		TW_BIND();
		current_thing = current_thing->next;
	}

	return 0;
}

