/////////////////////////////////////////////////////////////////////////////
//
// Web services CGI
//
// Copyright (c) 2011, ProSoft Technology, All Rights Reserved
//
// Author   : Henry Yu
//
// MODIFIED :
// 08/20/2011 HYU - Created
// 11/04/2011 HYU - Modified sendFirmwareData() to read the firmwaredata file with new format.
// 06/05/2013 PJB - Modified to add web services and show module information.
//
/////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <assert.h>
#include <signal.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/route.h>
#include <linux/sockios.h>
#include "cgiparse.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "mvi69api.h"


#define ENET_IF0_NAME "eth0"
#define ENET_IF1_NAME "eth1"

#ifndef stricmp
#define stricmp   strcasecmp
#endif

#define NETHELPER_EXIT_FAILURE -1
#define NETHELPER_EXIT_SUCCESS  0

// defines used by the rescue function
#define	BACKUP_SYSTEM_IMAGE	 		"/backup/systemrestore.tgz"
#define PSFT_UPDATE_FILE	 		"/psfttmp/update.tgz"
#define ALL_INIT_SCRIPTS	 		"/etc/init.d/*"
#define BACKUP_SYSTEM_RESTORE_CMD 	"tar -xzf /backup/systemrestore.tgz -C /"



void sendXMLHeader( void )
{
    // Print HTTP headers
    printf("HTTP/1.1 200 OK\r\n");
    printf("Expires: -1\r\n");
    printf("Cache-Control: no-cache, must-revalidate\r\n");
    printf("Pragma: no-cache\r\n");
    printf("Content-Type:application/xml;charset=iso-8859-1\r\n\r\n");
}

void sendHTMLHeader( void )
{
    // Print HTTP headers
    printf("HTTP/1.1 200 OK\r\n");
    printf("Expires: -1\r\n");
    printf("Cache-Control: no-cache, must-revalidate\r\n");
    printf("Pragma: no-cache\r\n");
    printf("Content-Type:text/html;charset=iso-8859-1\r\n\r\n");
}

void sendInvalidTransactionError( void )
{
    printf("HTTP/1.1 501 Invalid Transaction Requested\r\n");
    printf("Content-Type: text/plain\r\n\r\n");
    printf("Invalid Transaction Requested\n");
}

void sendEmptyResponse( void )
{
    // Print HTTP headers
    printf("HTTP/1.1 200 OK\r\n");
    printf("Content-Type: text/plain\r\n\r\n");
    printf("Transaction executed correctly\n");
}

///////////////////////////////////////////////////////////////////////////////

/*****************************************************************************
 * Purpose: Function that orders the linux system to shutdown
 *
 * Params: Nothing
 *
 * Returns: 0 if OK
 *
 * Comments:
 *
 *
******************************************************************************/
int RebootSystem( void )
{
    signal(SIGCLD, SIG_IGN);  // now I don't have to wait()
    pid_t pid;

    switch(pid=fork()) {
        case -1: // Something went wrong
            perror("fork");
            return -1;
        case 0:  // this is the child process
            //sleep(2); // Wait so at least this page can be returned
            // In order to be able to do a shutdown the process has to be root
            // because the web server doesn't run as root, the suid bit has to be added
            // to the plx80upgrade
            setuid(geteuid());
            // Kill init process
            kill(1,SIGTERM);
            exit(0);
        default: // This is the parent process
            return 0;
    }
    return(1);
}

/*****************************************************************************
 * Purpose: Function that determines if the module is running in setup mode
 *
 * Params: Nothing
 *
 * Returns: 0 if not in setup mode
 *          1 if setup jumper is installed
 *
 * Comments:
 *
******************************************************************************/
int inSetupMode( void )

{
	MVI69HANDLE 	handle;
	int 			result	= 0;

	//Open the CIP BP driver without BP access.
	if ( MVI69_OpenNB(&handle) == MVI69_SUCCESS )
	{
		MVI69_GetSetupJumper( handle, &result );

		MVI69_Close( handle );
	}

	return( result );
}



/*************************************************************************************************************
/ Purpose: Function that allows to read the IP address of a network interface
/
/ Params: char *ifname: Interface name that we want the information from. "eth0"
/              struct in_addr *ip: Buffer to hold the IP address
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int getIPAddress( char *ifname, struct in_addr *ip )
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);
	assert(ip != NULL);

	// Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
		memset(ip, 0, sizeof(struct in_addr));
 		return NETHELPER_EXIT_FAILURE;
 	}

	// Copy the interface name
	strncpy(reqIf.ifr_name, ifname, sizeof(reqIf.ifr_name));
	// Get the IP address via an ioctl
	if(ioctl(sockfd, SIOCGIFADDR , &reqIf) < 0)
	{
		perror("ioctl");
		memset(ip, 0, sizeof(struct in_addr));
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	else
	{
		memcpy(ip,&(((struct sockaddr_in *)&reqIf.ifr_addr)->sin_addr),sizeof(struct in_addr));
		close(sockfd);
		return NETHELPER_EXIT_SUCCESS;
	}
}

/*************************************************************************************************************
/ Purpose: Function that allows to read the MAC address of a network interface
/
/ Params: char *ifname: Interface name that we want the information from. "eth0"
/              unsigned char *buffer[6]: Buffer where the MAC address will be stored
/
/ Returns: 0 if successful
/          -1 on error
/
/ Comments:
/
***********************************************************************************************************/
int getMACAddress(char *ifname, unsigned char buffer[6])
{
	struct ifreq reqIf;
	int sockfd;

	assert(ifname != NULL);
	assert(buffer != NULL);

	// Create the Socket
	if ((sockfd = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
	{
		perror("socket");
		return NETHELPER_EXIT_FAILURE;
	}

	// Copy the interface name
	strncpy(reqIf.ifr_name,ifname,sizeof(reqIf.ifr_name));
	// Get the MAC address via an ioctl
	if(ioctl(sockfd, SIOCGIFHWADDR , &reqIf) < 0)
	{
		close(sockfd);
		return NETHELPER_EXIT_FAILURE;
	}
	else
	{
		int i;
		unsigned char *data = (unsigned char *)&reqIf.ifr_hwaddr.sa_data; // MAC Address data
		for(i=5; i>-1; i--)
		{
			buffer[i] = *data;
			data++;
		}
		close(sockfd);
		return NETHELPER_EXIT_SUCCESS;
	}
}


void shutdownTrn( void )
{
    // No input parameters in this transaction
    sendEmptyResponse();
    RebootSystem();
}

void sendIPAddress( void )
{
    struct in_addr e1Ip = { 0 };

    getIPAddress(ENET_IF0_NAME, &e1Ip);
    printf("IP:[\"%s\"]",inet_ntoa(e1Ip));
}

void sendMACAddress( void )
{
    unsigned char mac1Addr[6];

    getMACAddress(ENET_IF0_NAME, mac1Addr);

    printf("MAC:[\"%02X:%02X:%02X:%02X:%02X:%02X\"]",
        mac1Addr[5],mac1Addr[4],mac1Addr[3],mac1Addr[2],mac1Addr[1],mac1Addr[0]);
}


void sendUptime( void )
{
    double upTime,idleTime;
    int days,hours,mins,secs;
    int intUpTime;

    // Open the /proc/uptime file to get the total uptime
    FILE *procUptimeFile = fopen("/proc/uptime","r");
    // Return if the file cannot be opened
    if (procUptimeFile==NULL) return;
    // Only process this if we read two parameters from the uptime file
    if (fscanf(procUptimeFile,"%lf %lf",&upTime,&idleTime)==2)
    {
        printf("UPTIME:\"");
        intUpTime = upTime;
        days = intUpTime/86400;
        if (days > 0)
        {
            if (days>1) printf("%d days ",days);
            else printf("1 day ");
        }
        intUpTime = intUpTime % 86400;
        hours = intUpTime/3600;
        intUpTime = intUpTime % 3600;
        mins = intUpTime/60;
        intUpTime = intUpTime % 60;
        secs = intUpTime;
        printf("%02d:%02d:%02d\"",hours,mins,secs);
    }
    fclose(procUptimeFile);
}

void sendLinuxVersion( void )
{
    char versionString[301];

    char linuxStr[20];
    char versionStr[20];
    char version[20];
    char *pBuild;
    char *pBuildEnd;

    // Open the /proc/version to get the version information
    FILE *procVersionFile = fopen("/proc/version","r");
    // Return if the file cannot be opened
    if (procVersionFile==NULL) return;
    fgets(versionString,300,procVersionFile);
    // Parse the linux version number
    sscanf(versionString,"%s %s %s",linuxStr,versionStr,version);
    // Parse the Build Number
    pBuild = strstr(versionString,"#");
    pBuildEnd = strstr(pBuild," ");
    *pBuildEnd = 0;

    printf("LNXVERSION:\"%s %s\"",version,pBuild);
    fclose(procVersionFile);
}

void sendFirmwareData( void )
{
    char currentLine[301];
    char *pos;

    // Open the /psft/firmwaredata file to get the version information of the firmware file
    FILE *firmwareVersionFile = fopen("/psft/firmwaredata","r");
    // Return if the file cannot be opened
    if ( firmwareVersionFile == NULL )
    {
    }
    else
    {
    	while (fgets(currentLine,300,firmwareVersionFile)!=NULL)
    	{
    		// Remove \n from the strings
    		pos = strstr(currentLine,"\r\n");
    		if (pos!=NULL) *pos = 0;
    		pos = strstr(currentLine,"\n");
    		if (pos!=NULL) *pos = 0;

    		pos = strchr(currentLine, ':');
    		if (pos == NULL)
    		{
    			continue;
    		}

    		*pos = 0;
    		pos++;

    		printf("%s:\"%s\",", currentLine, pos);
    	}
    	fclose(firmwareVersionFile);
    }
}

void sendVersionInfo( void )
{
    char *				pos;
    MVI69HANDLE			handle;
    MVI69MODULEINFO		Module_Info;
    int 				result;
    char 				currentLine[301];
    char *				posStart = currentLine;


    sendLinuxVersion();
    printf(",");
    sendFirmwareData();

    FILE *firmwareVersionFile = fopen("/tmp/version","r");
    // Return N/A info if the file cannot be opened
    if (firmwareVersionFile==NULL)
    {
    	// if the file cannot be opened, read the info from the backplane

    	result = MVI69_OpenNB( &handle );
    	if ( result == MVI69_SUCCESS )
    	{
    		// read module information

    		result = MVI69_GetModuleInfo( handle, &Module_Info );
    		if ( result == MVI69_SUCCESS )
    		{
    			printf( "MODULE_NAME:\"MVI69E-LDM\","
    		            "MODULE_REVISION:\"N/A\","
    					"MODULE_SERIAL:\"%08lX\","
    		            "MODULE_STATUS:\"N/A\"",
    		            Module_Info.SerialNo );
    		}
    		else
    		{
    			printf( "MODULE_NAME:\"MVI69E-LDM\","
    					"MODULE_REVISION:\"N/A\","
    					"MODULE_SERIAL:\"failed to read\","
    					"MODULE_STATUS:\"N/A\"");
    		}
    		MVI69_Close( handle );
    	}
    	else
    	{
    		printf( "MODULE_NAME:\"N/A\","
    				"MODULE_REVISION:\"N/A\","
    				"MODULE_SERIAL:\"N/A\","
    				"MODULE_STATUS:\"N/A\"");
    	}
    }
    else
    {
        if (fgets(currentLine,300,firmwareVersionFile)!=NULL)
        {
            // Remove \n from the strings
            pos = strchr(currentLine, '{');
            if (pos!=NULL) posStart = pos + 1;

            pos = strrchr(currentLine, '}');
            if (pos!=NULL) *pos = 0;
        
            printf(posStart);
        }
        fclose(firmwareVersionFile);
    }
}


/******************************************************************************
 * Purpose: Function that provides the web browser with module info in JSON
 *          format
 *
 * Params:  NONE
 *
 *
 * Returns: Nothing
 *
 * Comments:
 *
 *****************************************************************************/
void getModuleInfo( void )
{
    sendHTMLHeader();
    printf("{");

    sendIPAddress();
    printf(",");
    sendMACAddress();
    printf(",");
    sendUptime();
    printf(",");
    sendVersionInfo();

    printf("}\n");
}


/////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////

int main(int argc, char *argv[])
{
    if(argc > 1)
    {
        if(argv[1][0] == '-' && (argv[1][1] == 'v' || argv[1][1] == 'V'))
        {
            return 0;
        }
    }

    int n = CGIParse_Parse();

    if (n == 0)
    {
        sendInvalidTransactionError();
        return 0;
    }

    const char * trn = CGIParse_GetParm("trn");


    if (stricmp(trn, "shutdown") == 0)
    {
    	shutdownTrn();
    }
    else if (stricmp(trn, "getmoduleinfo") == 0)
    {
    	getModuleInfo();
    }

    CGIParse_Release();

    return EXIT_SUCCESS;
}

