	TITLE	PRINTF - Output Formatting Routines For Assembly Programs.

;***	PRINTF -- Output Formatting Routines For Assembly Programs.
;
;1.	Functional Description.
;	This module contains the routines necessary to support the PRINTF
;	macro as defined in PRINTF.INC.  This code was originally written
;	for the General Software STARLITE project as a part of its kernel
;	debugger, and turned into a very useful tool for user MASM programs.
;
;2.	Modification History.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;3.	NOTICE: Copyright (C) 1990 General Software.
;
;4.	Build Environment.
;	MASM 5.10, no special switches.

TAB	=	09h			; define ASCII tab character code.
LF	=	0ah			; define ASCII line-feed character code.
CR	=	0dh			; define ASCII carriage-return code.
BEEP	=	07h			; define ASCII bell character code.
CTRLS	=	('S'-'A')		; define ASCII ^S character code.
SPACE	=	20h			; define ASCII blank character code.

	include ..\inc\usegs.inc
	include ..\inc\udefines.inc
	include ..\inc\umacros.inc
	include ..\inc\ustruc.inc

PRINTFCODE SEGMENT

HexTable	db	'0123456789abcdef'
SaveRet         dd	?
SaveStr         dw	?
SaveLocalRet	dw	?

;***	PrintfSvc - Print Formatted String.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is called by the PRINTF macro to print out a formatted
;	string, inserting arguments passed on the stack as appropriate. The
;	PRINTF macro call takes the following form:
;
;	PRINTF <string>,<op1,op2,op3,...,opn>
;
;	Where <string> is a print formatting specification similar to the
;	style of C's printf() function, and op1, op2, etc., are optional
;	arguments to be formatted. The following components can be specified
;	in the string:
;
;	Literal characters:
;	    \r	- carriage return.
;	    \n	- new line.
;	    \t	- tab.
;	    \b	- bell.
;	    \\	- backslash.
;	    \$	- dollar sign.
;
;	Formatting components (we use $ instead of C's % because MASM treats
;			       the '%' character as a special operation):
;	    $c	- format word argument as ASCII character, ignoring high half.
;	    $b	- format argument as hexadecimal byte, ignoring high half.
;	    $u	- format argument as unsigned word.
;	    $d	- format argument as signed word.
;	    $x	- format argument as hexadecimal word.
;	    $lu - format argument as unsigned longword (32 bit).
;	    $ld - format argument as signed longword (32 bit).
;	    $lx - format argument as hexadecimal longword (32 bit).
;	    $s	- format argument as ASCIIZ string.
;	    $s$ - format argument as $-terminated string.
;	    $s[n] - format argument as ASCIIZ string, maximum size n chars.
;	    NOTE: strings arguments require specifying offset & segment, i.e.:
;		  lea	dx, OFFSET DGROUP:Msg1
;		  PRINTF <The string is '$s'.\n>, <ds, dx>
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	[BP-0]	- FWA, first argument.
;	[BP-2]	- FWA, second argument, etc.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfSvc, PUBLIC, FAR

;	We are called with a FAR call, and the return address on the stack
;	points to a word containing an offset of the formatting string. So
;	we need that address, and also need to advance the return address
;	beyond that word, so do that now.

	pop	SaveRet.lo		; dump return address into SaveRet.
	pop	SaveRet.hi
	push	ds
	push	si
	lds	si, SaveRet		; (DS:SI) = FWA, format string word addr.
	mov	si, [si]		; (SI) = offset FWA, format string.
	mov	SaveStr, si		; keep in static for a moment.
	pop	si
	pop	ds
	add	SaveRet.lo, 2		; advance IP beyond the extra word.
	push	SaveRet.hi		; push return address back on stack.
	push	SaveRet.lo

;	Now all the caller's registers so we don't trash them.

	Pcall	SaveAll

;	Before entering the main loop, set up DS to point to the message
;	segment (PRINTFMSG) and reload (SI) to contain the offset address
;	of the print format string relative to PRINTFMSG.

	cld				; process strings in forward direction.
	mov	ax, PRINTFMSG
	mov	ds, ax
	ASSUME	DS:PRINTFMSG		; (DS) = message segment.
	mov	si, SaveStr		; (DS:SI) = FWA, format string.

PrintfSvc_Loop:
	lodsb				; (AL) = next char to print.
	or	al, al			; end of formatting string?
	jz	PrintfSvc_Exit		; yes. all done.
	cmp	al, '\'                 ; special literal character?
	je	PrintfSvc_Special	; yes.
	cmp	al, '$'                 ; formatting specification?
	je	PrintfSvc_Format	; yes.
	Pcall	PrintfPutCh		; put character in AL.
	jmp	SHORT PrintfSvc_Loop	; get next character.

;	Format special literal.

PrintfSvc_Special:
	lodsb				; (AL) = next character.
	or	al, al			; end of formatting string?
	je	PrintfSvc_Exit		; if so, just leave.
	Pcall	PrintfLiteralEscape	; do the work.
	jmp	SHORT PrintfSvc_Loop	; process more string.

;	Found a formatter for a variable.

PrintfSvc_Format:
	lodsb				; (AL) = format specification type.
	or	al, al			; end of formatting string?
	je	PrintfSvc_Exit		; if so, just leave.
	Pcall	PrintfFormat		; do the work.
	jmp	SHORT PrintfSvc_Loop	; process more string.

PrintfSvc_Exit:
	Pcall	RestoreAll		; restore all the registers.
EndProc PrintfSvc

;***	SaveAll - Save All Registers.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine saves all of the general registers on the stack and
;	returns to the caller.	While this is slower than just pushing
;	the registers, it makes the code smaller, which is more important
;	than output efficiency (the BIOS is 100x as slow as this routine).
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc SaveAll
	pop	SaveLocalRet		; save return address.
	push	ax
	push	bx
	push	cx
	push	dx
	push	si
	push	di
	push	ds
	push	es
	push	bp
	push	SaveLocalRet		; push IP back on stack.
EndProc SaveAll

;***	RestoreAll - Restore All Registers.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine restores all of the general registers on the stack
;	and returns to the caller.  While this is slower than just popping
;	the registers, it makes the code smaller, which is more important
;	than output efficiency (the BIOS is 100x as slow as this routine).
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc RestoreAll
	pop	SaveLocalRet		; save return address.
	pop	bp
	pop	es
	pop	ds
	pop	di
	pop	si
	pop	dx
	pop	cx
	pop	bx
	pop	ax
	push	SaveLocalRet		; push IP back on stack.
EndProc RestoreAll

;***	PrintfPutCh - Output Character Routine.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine writes one character to the screen with a BIOS/DOS call.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	AL	- character to write.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfPutCh
	Pcall	SaveAll                 ; save all general registers.

	IF	USE_BIOS

;	If FLOW_CONTROL is set as an option in PRINTF.INC, then we
;	call PrintfSuspendOutput, which will process ^S/^Q, which
;	the BIOS does not support.  If we're using DOS, then this
;	code doesn't get compiled, because DOS provides this functionality.

	IF	FLOW_CONTROL
	Pcall	PrintfSuspendOutput	; handle CTRL-S/CTRL-Q functions.
	ENDIF	; (FLOW_CONTROL)

	cmp	al, TAB                 ; is this a tab?
	jne	PrintfPutCh_Normal

;	We have a tab.	Because we don't have BIOS support, we just get
;	the current cursor location, add 8 to the X coordinate and mask
;	off the bottom three bits to get the new character position.
;	Subtracting the current position from the new position yields the
;	number of blanks we must write out to comprise this TAB.

	mov	ah, 3			; BIOS read cursor position.
	sub	bh, bh			; (BH) = page number 0.
	int	10h			; call BIOS.
	sub	dh, dh			; (DX) = current X coordinate.
	mov	cx, dx			; (CX) = saved X coordinate.
	add	cx, 8			; round up to next multiple of 8.
	and	cx, 11111000b		; mask off bottom 3 bits.
	sub	cx, dx			; (CX) = number of blanks first.
	or	cx, cx			; zero?
	jz	PrintfPutCh_Exit	; we're done.

PrintfPutCh_Loop:
	mov	ah, 0eh                 ; BIOS write text function.
	mov	al, SPACE		; write a blank.
	sub	bh, bh			; (BH) = page 0.
	int	10h			; print the blank.
	loop	PrintfPutCh_Loop	; do the rest.
	jmp	SHORT PrintfPutCh_Exit	; done.

PrintfPutCh_Normal:
	mov	ah, 0eh                 ; BIOS write text function.
	sub	bh, bh			; (BH) = page 0.
	int	10h			; call BIOS.

	ELSE	; (IF NOT USE_BIOS)

	mov	ah, 02h                 ; (AH) = write char function code.
	mov	dl, al			; (DL) = char to write.
	int	21h			; write character in (DL) to STDOUT.

	ENDIF	; (USE_BIOS)

PrintfPutCh_Exit:
	Pcall	RestoreAll		; restore general registers.
EndProc PrintfPutCh

;***	PrintfPutHexByte - Print Hex Byte.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine prints a hex byte in ASCII format on the screen.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	AL	- value to print.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfPutHexByte
	Pcall	SaveAll                 ; save general registers.
	xchg	ah, al
	mov	bx, ax			; (BX) = value to display.
	mov	cx, 2			; print 2 digits.

PrintfPutHexByte_Loop:
	push	cx
	mov	cl, 4			; shift count.
	rol	bx, cl			; position high nibble to bottom.
	mov	ax, bx			; (AX) = temporary value.
	and	ax, 000fh		; (AX) = single nibble to print.
	mov	si, ax			; (SI) = index into HexTable.
	mov	al, HexTable [si]	; (AL) = char to print.
	Pcall	PrintfPutCh		; print it out.
	pop	cx			; (CX) = restored loop count.
	loop	PrintfPutHexByte_Loop	; print rest of digits.

	Pcall	RestoreAll		; restore general registers.
EndProc PrintfPutHexByte

;***	PrintfPutHexWord - Print Hex Word.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine prints a hex word in ASCII format on the screen.
;	We reuse the PrintfPutHexByte code to save codespace.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	AX	- value to print.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfPutHexWord
	xchg	al, ah
	Pcall	PrintfPutHexByte
	xchg	ah, al
	Pcall	PrintfPutHexByte
EndProc PrintfPutHexWord

;***	PrintfPutUnsignedLong - Print Unsigned Longword.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine prints a 32-bit unsigned longword in ASCII format
;	on the screen.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	DX:AX	- 32-bit value to print.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfPutUnsignedLong
	Pcall	SaveAll                 ; save general registers.

;	If the value is zero, then just print a '0' and exit.  Otherwise,
;	engage the 32-bit divide routine.

	or	ax, ax			; is low half zero?
	jnz	PrintfPutUnsignedLong_DoIt ; nope.
	or	dx, dx			; is high half zero?
	jnz	PrintfPutUnsignedLong_DoIt ; nope.

;	We have a zero quantity, so just print a '0'.

	mov	al, '0'
	Pcall	PrintfPutCh
	jmp	SHORT PrintfPutUnsignedLong_Exit

;	Print a non-zero number.  Do that by dividing the 32-bit quantity
;	by a 16-bit number 10, recursively calling this routine to do the
;	same for the quotient, and after the recursive call returns, we
;	print the remainder digit.

PrintfPutUnsignedLong_DoIt:
	mov	di, dx
	mov	si, ax			; (DI:SI) = number to be divided.
	mov	cx, 10			; (CX) = divisor (10 for base 10).

;	Divide top part by 10 to get a quotient that we will pass to a
;	recursive call to PrintfPutUnsignedLong.  If the quotient is zero,
;	then don't recurse any more, because we're done.

	mov	ax, di
	sub	dx, dx			; (DX:AX) = high half of dividend.
	div	cx			; (AX) = quotient, (DX) = remainder.
	push	ax			; save high half of quotient.
	mov	ax, si			; (DX:AX) = low half of dividend.
	div	cx			; (AX) = quotient, (DX) = remainder.
	pop	dx			; (DX:AX) = 32-bit quotient.

	mov	bx, ax
	or	bx, dx			; is the 32-bit quotient zero?
	jz	@f			; if so, don't recurse any more.
	Pcall	PrintfPutUnsignedLong	; RECURSIVE CALL to display quotient.
@@:

;	Now divide our original number in (DI:SI) by 10 to get a remainder
;	that we will use to print our digit.  The remainder will be a number
;	in the range 0-9, because we are dividing by 10.

	mov	ax, di
	sub	dx, dx			; (DX:AX) = high half of dividend.
	div	cx			; (DX) = remainder.
	mov	ax, si			; (DX:AX) = low half of dividend.
	div	cx			; (DX) = remainder (actually, only DL).

;	Now print out the digit.

	mov	al, dl			; now print our digit.
	add	al, '0'                 ; (AL) = char to print.
	Pcall	PrintfPutCh		; write out current digit.

;	We're done.

PrintfPutUnsignedLong_Exit:
	Pcall	RestoreAll		; restore general registers.
EndProc PrintfPutUnsignedLong

;***	PrintfPutSignedLong - Print Signed Longword.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine prints a 32-bit signed longword in ASCII format on
;	the screen.  We reuse the code in PrintfPutUnsignedLong by negating
;	the value to be printed, printing a minus sign, and then we print
;	the actual longword.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	DX:AX	- value to print.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfPutSignedLong
	test	dx, 8000h		; is sign bit set in this dword?
	jz	@f			; if not, print as unsigned.

;	We have a negative number.  Convert it to a positive number and
;	print a minus sign before printing the number with unsigned code.

	push	ax
	mov	al, '-'
	Pcall	PrintfPutCh		; output minus sign.
	pop	ax

;	Subtract the quantity from zero, to get a negative version of the
;	32-bit number.

	push	bx
	push	cx
	sub	bx, bx
	sub	cx, cx			; (CX:BX) = 32-bit 0.
	sub	bx, ax			; subtract low parts.
	sbb	cx, dx			; subtract high parts with borrow.
	mov	dx, cx
	mov	ax, bx			; (DX:AX) = 32-bit quantity to print.
	pop	cx
	pop	bx

;	Now print the number in the 32-bit register pair (DX:AX).

@@:	Pcall	PrintfPutUnsignedLong	; display as longword.
EndProc PrintfPutSignedLong

;***	PrintfLiteralEscape - Print Literal Specification.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine writes the specified literal on the screen.
;
;	Literal characters:
;	    \r	- carriage return.
;	    \n	- new line.
;	    \t	- tab.
;	    \b	- bell.
;	    \\	- backslash.
;	    \$	- dollar sign.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	AL	- literal character specification.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfLiteralEscape
	cmp	al, 'r'                 ; carriage return?
	je	PrintfLiteralEscape_CR
	cmp	al, 'b'                 ; bell?
	je	PrintfLiteralEscape_Bell
	cmp	al, 't'                 ; tab?
	je	PrintfLiteralEscape_Tab
	cmp	al, 'n'                 ; new line?
	jne	PrintfLiteralEscape_OutputChar ; if not, print as normal char.

PrintfLiteralEscape_CRLF:
	mov	al, LF			; output LF first, then fall into
	Pcall	PrintfPutCh		; the CR output routine, below.

PrintfLiteralEscape_CR:
	mov	al, CR			; output CR.
	jmp	SHORT PrintfLiteralEscape_OutputChar

PrintfLiteralEscape_Bell:
	mov	al, BEEP		; output a BELL.
	jmp	SHORT PrintfLiteralEscape_OutputChar

PrintfLiteralEscape_Tab:
	mov	al, TAB                 ; output TAB character (fall into OutputChar).

;	Display the character in (AL).

PrintfLiteralEscape_OutputChar:
	Pcall	PrintfPutCh		; print ordinary character.
EndProc PrintfLiteralEscape

;***	PrintfLong - Format 32-bit Longword.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine prints a long word (32 bit operand) in signed,
;	unsigned, or hexadecimal format.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	AL	- secondary formatter ('u', 'x', or 'd').
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfLong
	inc	si			; by default, eat secondary formatter.

	cmp	al, 'u'                 ; display in unsigned format?
	je	PrintfLong_Unsigned	; if so.
	cmp	al, 'd'                 ; display in signed format?
	je	PrintfLong_Signed	; if so.
	cmp	al, 'x'                 ; display in hex format?
	je	PrintfLong_Hex		; if so.

;	We found an ordinary character following the $l.  That means we should
;	use a long signed format, and print the next character separately.

	dec	si			; don't eat secondary formatter.

;	We have an unsigned display format.

PrintfLong_Unsigned:
	mov	dx, [bp]		; (DX) = high 16-bit half to print.
	mov	ax, [bp-2]		; (AX) = low 16-bit half to print.
	sub	bp, 4			; pop 32-bit dword off stack frame.
	Pcall	PrintfPutUnsignedLong	; display as longword.
	ret

;	We have a signed display format.

PrintfLong_Signed:
	mov	dx, [bp]		; (DX) = high 16-bit half to print.
	mov	ax, [bp-2]		; (AX) = low 16-bit half to print.
	sub	bp, 4			; pop 32-bit dword off stack frame.
	Pcall	PrintfPutSignedLong	; display the number.
	ret

;	We have a hex display format.

PrintfLong_Hex:
	mov	ax, [bp]		; (AX) = high 16-bit half to print.
	Pcall	PrintfPutHexWord	; print hex value.
	mov	ax, [bp-2]		; (AX) = low 16-bit half to print.
	sub	bp, 4			; pop param off stack frame.
	Pcall	PrintfPutHexWord	; print hex value.
EndProc PrintfLong

;***	PrintfFormat - Print Format Specification With Argument.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine formats the next argument in the specified format.
;	If the formatting string is not correct, the results are unpredictable.
;
;	Formatting components:
;	    $c	- format argument as ASCII character (ignoring high half).
;	    $b	- format argument as hexadecimal byte (ignoring high half).
;	    $u	- format argument as unsigned word.
;	    $d	- format argument as signed word.
;	    $x	- format argument as hexadecimal word.
;	    $lu - format argument as unsigned longword (32 bit).
;	    $ld - format argument as signed longword (32 bit).
;	    $lx - format argument as hexadecimal longword (32 bit).
;	    $s	- format argument as ASCIIZ string.
;	    $s$ - format argument as $-terminated string.
;	    $s[n] - format argument as ASCIIZ string, maximum size n chars.
;	    NOTE: strings arguments require specifying offset & segment, ex:
;		  lea	dx, OFFSET DGROUP:Msg1
;		  PRINTF <The string is '$s'.\n>, <ds, dx>
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	AL	- format character.
;	DS:SI	- pointer to remainder of format specification, 0-byte terminated.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfFormat
	cmp	al, 'c'                 ; character specification?
	je	PrintfFormat_Char
	cmp	al, 'u'                 ; unsigned word?
	je	PrintfFormat_Unsigned
	cmp	al, 'd'                 ; signed word?
	je	PrintfFormat_Signed
	cmp	al, 'x'                 ; hex word?
	je	PrintfFormat_Hex
	cmp	al, 's'                 ; string?
	je	PrintfFormat_String
	cmp	al, 'b'                 ; hex byte?
	je	PrintfFormat_HexByte
	cmp	al, 'l'                 ; longword formatting?
	je	PrintfFormat_Long
	jmp	PrintfFormat_BadFormat	; if wrong formatting type.

PrintfFormat_Long:
	mov	al, [si]		; (AL) = secondary formatter.
	Pcall	PrintfLong		; process longword.
	ret

;	Print a raw character.

PrintfFormat_Char:
	mov	ax, [bp]		; (AX) = argument to print.
	Pcall	PrintfPutCh		; print AL.

;	Come here as a common exit point to pop off a 16-bit operand from
;	the stack.  This saves the SUB BP,2 and the RET statement, which
;	saves code space.

PrintfFormat_ExitPop:
	sub	bp, 2			; pop param off stack frame.
	ret

;	Print an unsigned 16-bit decimal number.

PrintfFormat_Unsigned:
	mov	ax, [bp]		; (AX) = argument to print.
	sub	dx, dx			; (DX:AX) = 32-bit number to print.
	Pcall	PrintfPutUnsignedLong	; print unsigned value.
	jmp	SHORT PrintfFormat_ExitPop

;	Print a signed 16-bit decimal integer.

PrintfFormat_Signed:
	mov	ax, [bp]		; (AX) = argument to print.
	cwd				; (DX:AX) = 32-bit signed number to print.
	Pcall	PrintfPutSignedLong	; print signed value.
	jmp	SHORT PrintfFormat_ExitPop

;	Print a 16-bit hexadecimal number.

PrintfFormat_Hex:
	mov	ax, [bp]		; (AX) = argument to print.
	Pcall	PrintfPutHexWord	; print hex value.
	jmp	SHORT PrintfFormat_ExitPop

;	Print an 8-bit hexadecimal number.

PrintfFormat_HexByte:
	mov	ax, [bp]		; (AL) = argument to print.
	Pcall	PrintfPutHexByte	; print hex value.
	jmp	SHORT PrintfFormat_ExitPop

;	Print a string.

PrintfFormat_String:
	mov	cx, 65535		; default limit on string size.
	cmp	byte ptr [si], 0	; is the next byte a zero?
	je	PrintfFormat_DumpAsciiz ; yes. use the default limit.
	cmp	byte ptr [si], '$'	; is it a $-terminated string?
	je	PrintfFormat_Dollar	; yes. do that specially.
	cmp	byte ptr [si], '['	; did he specify a string size?
	je	PrintfFormat_Bracket	; yes. go get string size.

PrintfFormat_DumpAsciiz:
	mov	dl, 0			; (DL) = zero-byte terminator.

PrintfFormat_DumpStr:
	mov	es, [bp]		; (ES) = seg FWA, string to print.
	mov	di, [bp-2]		; (ES:DI) = FWA, string to print.
	sub	bp, 4			; remove param from stack.
PrintfFormat_DumpStrLoop:
	mov	al, es:[di]		; (AL) = char to print.
	inc	di			; (DI) = FWA, next char after this one.
	cmp	al, dl			; have we found the terminator?
	je	PrintfFormat_Exit
	Pcall	PrintfPutCh		; write character out.
	loop	PrintfFormat_DumpStrLoop; do (CX) many characters at most.
PrintfFormat_Exit:
	ret				; finished.

PrintfFormat_Dollar:
	lodsb				; eat dollar sign format specifier.
	mov	dl, '$'                 ; terminate with dollar sign.
	jmp	SHORT PrintfFormat_DumpStr ; go dump out the string.

PrintfFormat_Bracket:
	lodsb				; eat bracket.
	sub	ah, ah			; for 16-bit adds.
	sub	cx, cx			; (CX) = count.
	mov	dh, 10			; multiplier.
PrintfFormat_LengthLoop:
	lodsb				; (AL) = length.
	or	al, al			; end of string?
	jz	PrintfFormat_BadFormat	; formatting error, no close bracket.
	cmp	al, ']'                 ; closing bracket?
	je	PrintfFormat_DumpAsciiz ; yes. found it.
	sub	al, '0'
	xchg	ax, cx			; (AX) = sum, (CX) = char.
	mul	dh			; (AX) = sum*10.
	add	ax, cx			; (AX) = sum*10 + char.
	xchg	ax, cx			; (AX) = char, (CX) = sum.
	jmp	SHORT PrintfFormat_LengthLoop ; get more chars.

;	Bad formatting code.  Display '[***]'.

PrintfFormat_BadFormat:
	mov	al, '['
	Pcall	PrintfPutCh
	mov	al, '*'
	Pcall	PrintfPutCh
	Pcall	PrintfPutCh
	Pcall	PrintfPutCh
	mov	al, ']'
	Pcall	PrintfPutCh
EndProc PrintfFormat

;***	PrintfSuspendOutput - Handle CTRL-S/CTRL-Q Processing.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is called by output routines in the PRINTF package
;	to handle user flow control requests in the form of ^S/^Q characters.
;	If you don't want this functionality, then set FLOW_CONTROL = 0 in
;	the PRINTF.INC file.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	90/11/09.	OEM DOS Release for COMDEX.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc PrintfSuspendOutput

;	We only enable flow control if USE_BIOS is in effect, because
;	the DOS function automatically provides ^S/^Q functionality.

	IF	USE_BIOS
	IF	FLOW_CONTROL

	Pcall	SaveAll                 ; save all general registers.

PrintfSuspendOutput_Loop:
	mov	ah, 01h                 ; peek character function.
	int	16h			; call BIOS.
	jz	PrintfSuspendOutput_Exit; no char ready.

;	A character is available.  If it is Ctrl-S, then suspend output.

PrintfSuspendOutput_CharAvail:
	cmp	al, CTRLS		; is this a CTRL-S?
	jne	PrintfSuspendOutput_Exit; nope, just leave it in the buffer.

;	He pressed CTRL-S, so wait for the next char to continue.

	sub	ah, ah			; get CTRL-S char.
	int	16h			; call BIOS.
	sub	ah, ah			; get CTRL-Q char.
	int	16h
	jmp	SHORT PrintfSuspendOutput_Loop; get more chars if fast typematic.

PrintfSuspendOutput_Exit:
	Pcall	RestoreAll		; restore general registers.

	ENDIF	; (FLOW_CONTROL)
	ENDIF	; (USE_BIOS)

EndProc PrintfSuspendOutput

PRINTFCODE ENDS

	END
