/**  $Revision:   1.0  $
/**/
/************************************************************************
*
*   Title:  Sample barcode-reader application for the 1756-MVI Module
*
*   Abstract:
*
*   This sample program implements a simple interface to a barcode reader
*   or similar device.  It uses the CIP API and transfers data
*   using scheduled connected data.  (This scheme provides greater
*   throughput than unshceduled messaging using the CIP Generic instruction.)
*
*   Note: This sample program performs the same function as the MVI API 
*   sample program BARCD_IO.C.  The same ladder code will work for either
*   program.
*
*   This sample also illustrates the use of multiple threads.  One thread
*   is used to transmit serial data, and another thread is used to receive
*   serial data.
*
*   Environment:    1756-MVI Communications Module
*                   CIP API
*                   General Software DOS 6-XL
*                   Borland/Microsoft C/C++ Compiler (16-bit)
*
*   Author:         JMC
*                   Copyright (c) 1999-2000 Online Development, Inc.
*   
*
*   Notes:
*
*   In this program, the "Input Image" refers to the data that the MVI module is
*   transmitting to the 5550 controller.  The "Output Image" refers to data that
*   the 5550 controller is transmitting to the MVI module.
*
*
*   Receiving Serial Data
*   ---------------------
*
*   Serial data received from an enabled port will be written to
*   the input image.  The first word of the input data contains the port
*   number from which the data was received (0=PRT1, 1=PRT2, 2=PRT3).
*   The second word of the input data contains the number of bytes
*   of serial data being transferred.  The last two words of the input
*   image are used for handshaking with the controller.  This leaves
*   246 words (492 bytes) available for serial data.
*
*   The termination character is transferred.  Unused bytes at
*   the end of the message buffer are set to zero.
*
*   MVI Input Data Usage (data received from MVI module)
*   --------------------
*
*   Word   0: Port number (0=PRT1, 1=PRT2, 2=PRT3)
*   Word   1: Length of data (1 - 492 bytes)
*   Word   2: First word of serial data
*      . . .
*   Word 247: Last word of serial data
*   Word 248: Input data sequence
*   Word 249: Output acknowledge
*
*   Note: To insure data block integrity over the input image, it is
*   important that the handshake word (input data sequence) is located
*   after the data block in the image.
*
*
*   Transmitting Serial Data
*   ------------------------
*
*   Messages received from the controller will be sent to the port
*   specified.  The first two words of the output data are used for
*   handshaking between the controller and the MVI module.  
*   The third word of the output data contains the port
*   number from which the data was received (0=PRT1, 1=PRT2, 2=PRT3).
*   The fourth word of the output data contains the number of bytes
*   of serial data being transferred.  This leaves
*   244 words (488 bytes) available for serial data.
*
*   MVI Output Data Usage (data sent to MVI module)
*   --------------------
*
*   Word   0: PLC Status
*   Word   1: reserved
*   Word   2 (0 on 5550): Output sequence
*   Word   3 (1 on 5550): Input acknowledge
*   Word   4 (2 on 5550): Port number (0=PRT1, 1=PRT2, 2=PRT3)
*   Word   5 (3 on 5550): Length of data (1 - 488 bytes)
*   Word   6 (4 on 5550): First word of serial data
*      . . .
*   Word 249 (247 on 5550): Last word of serial data
*
*   Note: To insure data block integrity over the output image, it is
*   important that the handshake word (output data sequence) is located
*   before the data block in the image.
*
*
*
*   Using the program
*   -----------------
*
*   The program allows the baud rate and string termination character
*   to be specified on the command line.  It also allows either or
*   both serial ports to be specified.  For simplicity, the baud and
*   termination character parameters are applied to all serial ports.
*   The ports are setup for 8 bits, no parity, and 1 stop bit.
*
*   Syntax: 
*
*         bar_caio [-baud=index] [-tc=term char] -p[1|2|3]
*
*   Options:
*
*         -baud=[index]     Specifies serial port baud rate.  Default
*                           is 9600 (index=7).
*
*         -tc=[term char]   Specifies termination character.  ASCII
*                           value of character in decimal is expected.
*                           Default is 10 (line feed).
*
*   At least one of the following must be specified:
*
*         -p1               Indicates that PRT1 is enabled
*
*         -p2               Indicates that PRT2 is enabled
*
*         -p3               Indicates that PRT3 is enabled
*                   
* 
************************************************************************/


/*=======================================================================
=                           INCLUDE FILES                               =
=======================================================================*/

#include    <stdio.h>
#include    <stdlib.h>
#include    <conio.h>
#include    <string.h>
#include    "kernel.h"      // definitions to support multithreading
#include    "cipapi.h"      // CIP API definitions
#include    "mvispapi.h"    // Serial Port API definitions


//#define DEBUG

/*=======================================================================
=                    MODULE WIDE DEFINITIONS                            =
=======================================================================*/

/*
 * IO Image Offset Definitions (word offsets)
 */

#define INP_PORT_OFFSET     0           // Serial port number of received data
#define INP_LEN_OFFSET      1           // Length (in bytes) of received data
#define INP_DATA_OFFSET     2           // First word of received data
#define INP_SEQ_OFFSET      248         // MVI to PLC data sequence handshake
#define INP_ACK_OFFSET      249         // PLC to MVI data acknowledge

#define OUT_SEQ_OFFSET      2           // PLC to MVI data sequence handshake
#define OUT_ACK_OFFSET      3           // MVI to PLC data acknowledge
#define OUT_PORT_OFFSET     4           // Serial port number for data
#define OUT_LEN_OFFSET      5           // Length (in bytes) of data to send
#define OUT_DATA_OFFSET     6           // First word of data to send

#define MAX_DATA_SZ         488   // Maximum string length that can be transferred

/*
 * Instance numbers implemented
 */
#define DATA_INPUT_INSTANCE      1
#define DATA_OUTPUT_INSTANCE     2
#define STATUS_INPUT_INSTANCE    5
#define STATUS_OUTPUT_INSTANCE   6

/*
 * Instance sizes
 */
#define DATA_INPUT_MAXSIZE     500      // max size in bytes
#define DATA_OUTPUT_MAXSIZE    500      // max size in bytes
#define STATUS_INPUT_MAXSIZE     6      // max size in bytes
#define STATUS_OUTPUT_MAXSIZE    0      // max size in bytes

// Serial Port Control Structure
typedef struct
{
    int  Enable;                // 1 means port is enabled for barcode data
    int  ComPort;               // Set to COM1, COM2, or COM3
    BYTE TermChar;              // termination character
    WORD StringLength;          // Max string length
    WORD InBufidx;              // Current receive data buffer pointer
    BYTE InBuf[MAX_DATA_SZ+1];  // Message buffer
} PORTCFG;


/*=======================================================================
=                      LOCAL SUPPORT PROTOTYPES                         =
=======================================================================*/

void ErrorExit(int);
void servicePort(PORTCFG *);
THREAD __loadds checkTxMsg(void);   // thread to handle messages from controller

/*=======================================================================
=                    MODULE WIDE GLOBAL VARIABLES                       =
=======================================================================*/

PORTCFG Port1Cfg;           // Allocate data for port 1
PORTCFG Port2Cfg;           // Allocate data for port 2
PORTCFG Port3Cfg;           // Allocate data for port 3
int RunThread = 1;          // Used by checkTxMsg to exit if necessary
MVIHANDLE handle;           // Handle returned from MVIcip_Open function
MVIHANDLE objHandle;        // Handle returned from MVIcip_RegisterAssemblyObj

// Data connection
MVIHANDLE dataConnHandle=0;                 // Handle for data connection
BYTE dataConnOpen=0;                        // 1 if data connection is open
WORD dataInputSize=0;                       // Set in connect_proc
WORD dataOutputSize=0;                      // Set in connect_proc
BYTE dataInputBuf[DATA_INPUT_MAXSIZE];      // Data input buffer
BYTE dataOutputBuf[DATA_OUTPUT_MAXSIZE];    // Data output buffer

// Status connection
MVIHANDLE statusConnHandle=0;               // Handle for status connection
BYTE statusConnOpen=0;                      // 1 if status connection is open
WORD statusInputSize=0;                     // Set in connect_proc
WORD statusOutputSize=0;                    // Set in connect_proc
BYTE statusInputBuf[STATUS_INPUT_MAXSIZE];  // Status input buffer


/*=======================================================================
=                       LOCAL SUPPORT ROUTINES                          =
=======================================================================*/

/************************************************************************
*
*     Entry point:                                                      
*       ErrorExit                                           
*
*     Description:                                                      
*       This routine is called whenever an error code is returned from
*       an API routine.  It displays the message associated with the
*       error, then exits the program.
*       
*     Arguments:                                                        
*
*       errcode             : int                           ( input )
*         Error return value from a backplane API function
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void ErrorExit(int errcode)
{
    char errbuf[80];

    RunThread = 0;                  // signal child thread to exit
    PassTimeSlice();                // give child thread time to exit

    if (errcode)
    {
        MVIcip_ErrorString(errcode, errbuf);   // get error message
        printf("%s\n", errbuf);
    }

    MVIcip_Close(handle);           // should always close before exiting
    exit(1);
}


/************************************************************************
*
*     Entry point:                                                      
*       usage
*
*     Description:                                                      
*       This routine is called when invalid command-line options
*       are encountered.
*       
*     Arguments:                                                        
*
*       name                : char *                        ( input )
*         program name string
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void usage(char *name)
{
    printf("\nUsage: %s [-baud=index] [-tc=term ch] [-setup] -p[1|2|3]", name);
    printf("\nOptions:");
    printf("\n   -baud=index     index specifies baud rate (default=7 or 9600 baud)");
    printf("\n   -tc=term ch     termination character value in decimal (default=10)");
    printf("\n   -setup          do not exit if setup jumper installed");
    printf("\nAt least one of the following must be specified:");
    printf("\n   -p1             enable COM1");
    printf("\n   -p2             enable COM2");
    printf("\n   -p3             enable COM3");
}


/*
** Stack Checking must be disabled for callback routines !!
*/
#ifdef __BORLANDC__
#pragma option -N-
#endif
#ifdef _MSC_VER
#pragma check_stack (off)
#endif

/************************************************************************
*
*     Entry point:                                                      
*       connect_proc                                           
*
*     Description:                                                      
*       Class 1 connection callback routine.  This routine is called when
*       a forward open or close is received for the registered assembly
*       object.
*
*       The following Class 1 connection points (instances) are supported:
*
*       1 - Input data
*       2 - Output data
*       5 - Status input
*       6 - Status output (0 length)
*
*       Note: The Input data and Output data connections are not used by
*       this sample application.  They are supported here as illustration
*       and to maintain compatibility with the MVI API version of this
*       application.
*
*     Arguments:                                                        
*
*       objHandle           : MVIHANDLE                     ( input )
*         Object handle from MVIcip_RegisterAssemblyObj
*
*       psConn              : MVICIPCONNSTRUC *             ( input )
*         Pointer to a structure containing information from the
*         forward open/close.
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par*
#endif
MVICALLBACK connect_proc(
    MVIHANDLE objHandle,
    MVICIPCONNSTRUC *psConn )
{

    switch( psConn->reason )        // Open, Open Complete, or Close
    {
        case MVI_CIP_CONN_OPEN:                 // Open Request

            // Check for data connection
            if ( (psConn->producerCP == DATA_INPUT_INSTANCE) &&
                 (psConn->consumerCP == DATA_OUTPUT_INSTANCE) )
            {
                // Only one connection to be active at a time
                if( dataConnHandle != (MVIHANDLE) NULL )
                {
                    *psConn->extendederr = MVI_CIP_EX_CONNECTION_USED;
                    return( MVI_CIP_FAILURE );
                }

                // Check size
                if( (psConn->txDataSize > DATA_INPUT_MAXSIZE) ||
                    (psConn->rxDataSize > DATA_OUTPUT_MAXSIZE) )
                {
                   *psConn->extendederr = MVI_CIP_EX_BAD_SIZE;
                   return( MVI_CIP_FAILURE );
                }

                // Everything checks out - accept the connection
                // Save the connection handle and other info
                dataConnHandle = psConn->connHandle;
                dataInputSize = psConn->txDataSize;
                dataOutputSize = psConn->rxDataSize;
            }

            // Check for status Connection
            else if ( (psConn->producerCP == STATUS_INPUT_INSTANCE) &&
                      (psConn->consumerCP == STATUS_OUTPUT_INSTANCE) )
            {
                // Only one connection to be active at a time
                if( statusConnHandle != (MVIHANDLE) NULL )
                {
                    *psConn->extendederr = MVI_CIP_EX_CONNECTION_USED;
                    return( MVI_CIP_FAILURE );
                }

                // Check size
                if( (psConn->txDataSize > STATUS_INPUT_MAXSIZE) ||
                    (psConn->rxDataSize > STATUS_OUTPUT_MAXSIZE) )
                {
                    *psConn->extendederr = MVI_CIP_EX_BAD_SIZE;
                    return( MVI_CIP_FAILURE );
                }

                // Accept the connection
                // Save status connection handle
                statusConnHandle = psConn->connHandle;
                statusInputSize = psConn->txDataSize;

            }
            else    // Unknown connection point
            {
                return( MVI_CIP_FAILURE );
            }
            break;

        case MVI_CIP_CONN_OPEN_COMPLETE:        // Open Complete
            /*
            ** The connection is now up and running. If desired, load
            ** initial transmit data now. Also now is the time to notify
            ** the application threads that the connection is running.
            */
            if ( psConn->connHandle == dataConnHandle )
            {
                // Load initial transmit data
                MVIcip_WriteConnected( handle, dataConnHandle, dataInputBuf, 
                  0, dataInputSize );
                dataConnOpen = 1;          // indicate connection is now open
            }
            else if ( psConn->connHandle == statusConnHandle )
            {
                statusConnOpen = 1;        // indicate connection is now open
            }

            break;

        case MVI_CIP_CONN_CLOSE:                // Connection Closed
            /*
            ** The connection referenced by the connHandle has been
            ** closed.
            */
            if ( psConn->connHandle == dataConnHandle )
            {
                /*
                ** Here we need to:
                **    Free any allocated resources
                **    Clear handles
                */
                dataConnOpen = 0;           // indicate connection is now closed 
                dataConnHandle = (MVIHANDLE) NULL;
            }
            else if ( psConn->connHandle == statusConnHandle )
            {
                statusConnOpen = 0;         // indicate connection is now closed 
                statusConnHandle = (MVIHANDLE) NULL;
            }

            break;

        default:
            return( MVI_CIP_FAILURE );
    }

    return( MVI_SUCCESS );

} /* end connect_proc() */
#ifdef __BORLANDC__
#pragma warn .par*
#endif


/************************************************************************
*
*     Entry point:                                                      
*       service_proc                                           
*
*     Description:                                                      
*       Class 3 service callback routine.  This routine is called when
*       an unscheduled Class 3 service request is received.
*
*       This application supports the following Class 3 instances (called 
*       'Object ID' in the 5550 message configuration dialog)
*       of the assembly object (class code/object type 4):
*
*       1 - Input data (get only)
*       2 - Output data (get & set)
*       5 - Status input (get only)
*       7 - Message input data (get only)
*       8 - Message output data (get & set)
*
*       The following services are supported:
*
*       Get Attribute Single: 0x0E
*       Set Attribute Single: 0x10
*
*       The following attributes are supported:
*
*       Data: 0x03
*       Size: 0x04
*
*       There are two Class 3 messages that are used to implement 
*       MVI Messaging:
*
*           Send a message to the MVI module
*               - attribute 3 (data)
*               - instance 8 (msg output data)
*               - service 0x10 (set)
*
*           Read a message from the MVI module
*               - attribute 3 (data)
*               - instance 7 (msg input data)
*               - service 0x0E (get)
*
*     Arguments:                                                        
*
*       objHandle           : MVIHANDLE                     ( input )
*         Object handle from MVIcip_RegisterAssemblyObj
*
*       psServ              : MVICIPSERVSTRUC *             ( input )
*         Pointer to a structure containing information from the
*         service request.
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       MVI_SUCCESS
*       MVI_CIP_ATTR_NOT_SETTABLE
*       MVI_CIP_PARTIAL_DATA
*       MVI_CIP_BAD_SIZE
*       MVI_CIP_NO_RESOURCE 
*       MVI_CIP_BAD_SERVICE
*       MVI_CIP_BAD_INSTANCE 
*       MVI_CIP_BAD_ATTR_DATA
*       MVI_CIP_BAD_ATTR
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
#ifdef __BORLANDC__
#pragma warn -par*
#endif
MVICALLBACK service_proc(
    MVIHANDLE objHandle,
    MVICIPSERVSTRUC *psServ )
{

    /*
    ** Process according to the instance attribute.
    */
    switch( psServ->attribute )
    {
        case MVI_CIP_IA_DATA:

            /*
            ** Check the Service code.
            */
            switch( psServ->serviceCode )
            {
                case MVI_CIP_SC_SET_ATTR_SINGLE:

                    if( *psServ->msgSize != 0 )
                    {
                        switch( psServ->instance )
                        {
                            // Do not allow writes to any of these
                            case DATA_INPUT_INSTANCE:
                            case STATUS_INPUT_INSTANCE:
                            case STATUS_OUTPUT_INSTANCE:
                                return(MVI_CIP_ATTR_NOT_SETTABLE);

                            // Allow write to output data
                            case DATA_OUTPUT_INSTANCE:
                                if( *psServ->msgSize != dataOutputSize )
                                {
                                    return( MVI_CIP_PARTIAL_DATA );
                                }
                                memcpy(dataOutputBuf, *psServ->msgBuf, dataOutputSize);
                                break;

                            default:
                                return( MVI_CIP_BAD_INSTANCE );

                        } /* end switch( instance ) */
                    }
                    break;

                case MVI_CIP_SC_GET_ATTR_SINGLE:
                    /*
                    ** Return the data for the instance specified.
                    */
                    switch( psServ->instance )
                    {
                        case DATA_INPUT_INSTANCE:
                            *psServ->msgBuf   = dataInputBuf;
                            *psServ->msgSize  = dataInputSize;
                            break;

                        case DATA_OUTPUT_INSTANCE:
                            *psServ->msgBuf   = dataOutputBuf;
                            *psServ->msgSize  = dataOutputSize;
                            break;

                        case STATUS_INPUT_INSTANCE:
                            *psServ->msgBuf   = statusInputBuf;
                            *psServ->msgSize  = statusInputSize;
                            break;
                            
                        default:
                            return( MVI_CIP_BAD_INSTANCE );

                    } /* end switch( instance ) */

                    break;

                case MVI_CIP_SC_GET_MEMBER:
                case MVI_CIP_SC_SET_MEMBER:
                default:
                    return( MVI_CIP_BAD_SERVICE );

            } /* end switch( serviceCode ) */
            break;

        case MVI_CIP_IA_SIZE:

            /*
            ** Data size.
            ** Disallow anything but get attribute access.
            */

            /*
            ** Check the Service code.
            */
            switch( psServ->serviceCode )
            {
                case MVI_CIP_SC_SET_ATTR_SINGLE:
                    return( MVI_CIP_ATTR_NOT_SETTABLE );

                case MVI_CIP_SC_GET_ATTR_SINGLE:
                    /*
                    ** Return the size for the instance specified.
                    */
                    switch( psServ->instance )
                    {
                        case DATA_INPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&dataInputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case DATA_OUTPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&dataOutputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case STATUS_INPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&statusInputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        case STATUS_OUTPUT_INSTANCE:
                            *psServ->msgBuf   = (BYTE *)&statusOutputSize;
                            *psServ->msgSize = sizeof(WORD);
                            break;
                        default:
                            return( MVI_CIP_BAD_INSTANCE );
                    } /* end switch( instance ) */

                    break;

                case MVI_CIP_SC_GET_MEMBER:
                case MVI_CIP_SC_SET_MEMBER:
                    return( MVI_CIP_BAD_ATTR_DATA );

                default:
                    return( MVI_CIP_BAD_SERVICE );

            } /* end switch( serviceCode ) */
            break;

        case MVI_CIP_IA_NUM_MEMBERS:
        case MVI_CIP_IA_MEMBER_LIST:
        default:
            return( MVI_CIP_BAD_ATTR );

    } /* end switch( attribute ) */

    return( MVI_SUCCESS );

} /* end service_proc() */
#ifdef __BORLANDC__
#pragma warn .par*
#endif

/*
** Return Stack Checking to its default state
*/
#ifdef __BORLANDC__
#pragma option -N.
#endif
#ifdef _MSC_VER
#pragma check_stack ()
#endif


/*=======================================================================
=                       MAIN ENTRY POINT                                =
=======================================================================*/

/************************************************************************
*
*     Entry point:                                                      
*       main                                           
*
*     Description:                                                      
*
*     Arguments:                                                        
*       none
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       none
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
*     debugging/error printf's will only be seen if console is enabled.
*
************************************************************************/
void main(int argc, char *argv[])
{
    char *arg;
    int rc;
    int n;
    int mode;
    BYTE Baud = BAUD_9600;      // default baud rate
    BYTE TermChar = _LF;        // default termination character
    int Setup = 0;
    WORD timeout;

    // init port config structs
    memset(&Port1Cfg, 0, sizeof(PORTCFG));
    memset(&Port2Cfg, 0, sizeof(PORTCFG));
    memset(&Port3Cfg, 0, sizeof(PORTCFG));

    // Initialize input image (Tx data)
    memset(dataInputBuf, 0, sizeof(dataInputBuf));

    // parse the command line options
    for (n=1; n<argc; n++) {    // scan for arguments
        arg = argv[n];

		if (strnicmp("-baud=", arg, 6) == 0) {
            Baud = (BYTE) atoi(arg+6);
            if (Baud > BAUD_115200)
            {
                usage(argv[0]);
                exit(1);
            }
            continue;
        }

		if (strnicmp("-tc=", arg, 4) == 0) {
            TermChar = (BYTE) atoi(arg+4);
            continue;
        }

		if (strnicmp("-setup", arg, 6) == 0) {
            Setup = 1;
            continue;
        }

		if (strnicmp("-p1", arg, 3) == 0) {
            Port1Cfg.Enable = 1;
            continue;
        }

		if (strnicmp("-p2", arg, 3) == 0) {
            Port2Cfg.Enable = 1;
            continue;
        }

		if (strnicmp("-p3", arg, 3) == 0) {
            Port3Cfg.Enable = 1;
            continue;
        }

        usage(argv[0]);
        exit(1);
    }

    // Make sure at least one port is enabled
    if ((!Port1Cfg.Enable) && (!Port2Cfg.Enable) && (!Port3Cfg.Enable))
    {
        usage(argv[0]);
        exit(1);
    }

    // Open the Backplane API
    if (MVI_SUCCESS != (rc = MVIcip_Open(&handle)))
    {
        printf("\nMVIcip_Open failed: %d\n", rc);
        ErrorExit(rc);
    }

    // Check for setup mode - if so, exit now unless setup option is present.
    // The purpose for this is to allow the user to regain control of the
    // module by installing the Setup Jumper.  In a typical configuration,
    // the module application is run from AUTOEXEC.BAT.  In this case,
    // we exit the application if the Setup Jumper is installed, unless
    // the -setup command line option is preset.  This option is useful
    // for application debugging.
    MVIcip_GetSetupMode(handle, &mode);
    if (!Setup && mode)
    {   // Print the banner and version only if setup jumper is on
        printf("\nMVI Barcode Sample Application (CIP API / connected IO example)");
        printf("\nCopyright (c) 2000 Online Development, Inc.\n");
        printf("\nSetup jumper installed: exiting application\n");
        MVIcip_Close(handle);
        exit(0);
    }

    // Register the assembly object
    if (MVI_SUCCESS != (rc = MVIcip_RegisterAssemblyObj( handle, &objHandle,
                                0L, connect_proc, service_proc, NULL )))
    {
        printf("\nMVIcip_RegisterAssemblyObj failed: %d\n", rc);
        ErrorExit(rc);
    }

    // Initialize the serial port(s)
    if (Port1Cfg.Enable)
    {
        rc = MVIsp_Open(COM1, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port1Cfg.ComPort = COM1;
        Port1Cfg.StringLength = MAX_DATA_SZ - 1;
        Port1Cfg.TermChar = TermChar;
    }
    if (Port2Cfg.Enable)
    {
        rc = MVIsp_Open(COM2, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port2Cfg.ComPort = COM2;
        Port2Cfg.StringLength = MAX_DATA_SZ - 1;
        Port2Cfg.TermChar = TermChar;
    }
    if (Port3Cfg.Enable)
    {
        rc = MVIsp_Open(COM3, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port3Cfg.ComPort = COM3;
        Port3Cfg.StringLength = MAX_DATA_SZ - 1;
        Port3Cfg.TermChar = TermChar;
    }

    // Wait for the 5550 to open the data connection
    timeout = 100;
    while(!dataConnOpen && timeout--)
    {
        MVIcip_Sleep(handle, 100);
    }
    if (!dataConnOpen)
    {
        printf("\nConnections never opened.  Check 5550.\n");
        MVIcip_Close(handle);
        exit(1);
    }
#ifdef DEBUG
    printf("\nData connection open: Input size = %d bytes, Output size = %d bytes\n",
            dataInputSize, dataOutputSize );
#endif

    // Start the thread to handle messages from the controller
    AllocateThreadLong(checkTxMsg, 0, THREAD_PRIORITY_DEFAULT);

    // Main Loop - never exits - handles data received from serial ports
#ifdef DEBUG
    while(!kbhit())     // handy for testing - just don't use -p1 option
#else
    while(1)
#endif
    {
        servicePort(&Port1Cfg);     // service rx data
        servicePort(&Port2Cfg);     // service rx data
        servicePort(&Port3Cfg);     // service rx data
        PassTimeSlice();            // give the checkTxMsg thread a chance to run
    }

// The following lines are needed if the loop above is allowed to exit:
#ifdef DEBUG
    RunThread = 0;              // signal child thread to exit
    MVIcip_Sleep(handle, 20);   // give child thread time to exit
    MVIcip_Close(handle);
#endif

}


/************************************************************************
*
*     Entry point:                                                      
*       servicePort                                           
*
*     Description:                                                      
*       Check for data from the serial port.  If data is ready, move
*       it to the port's buffer.  If the message is complete (either
*       by matching the termination character or the string length),
*       then send the data to the PLC.
*
*     Arguments:                                                        
*
*       portcfg             : PORTCFG *                     ( input )
*         Pointer to port control structure
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void servicePort(PORTCFG *portcfg)
{
    int rc;
    int len;
    WORD w;
    static WORD inputSeq = 0;
    int timeout;

    // Return if port is not enabled 
    if (!portcfg->Enable)
        return;

    // Calculate number of characters left to go
    len = portcfg->StringLength - portcfg->InBufidx;

    rc = MVIsp_Gets(portcfg->ComPort, &portcfg->InBuf[portcfg->InBufidx],
             portcfg->TermChar, &len, TIMEOUT_ASAP);

    if (len == 0)                       // Return if no data in queue
        return;

    // Update buffer index
    portcfg->InBufidx += len;

    // Termination test - see if string is complete
    if (rc == MVI_SUCCESS)
    {
        // String is complete - build a message and send to PLC 

        // Blink LED for debugging
        MVIcip_SetUserLED(handle, MVI_LED_USER1, MVI_LED_STATE_ON);

        // Write port number to buffer
        *(WORD *)&dataInputBuf[INP_PORT_OFFSET*2] = (WORD) portcfg->ComPort;

        // Write string length to buffer
        *(WORD *)&dataInputBuf[INP_LEN_OFFSET*2] = (WORD) portcfg->InBufidx;

        // Write string to data buffer
        memcpy(&dataInputBuf[INP_DATA_OFFSET*2], portcfg->InBuf, portcfg->InBufidx);

        // Now bump the input data sequence number
        inputSeq++;
        *(WORD *)&dataInputBuf[INP_SEQ_OFFSET*2] = inputSeq;

        // Update the Tx data
        rc = MVIcip_WriteConnected(handle, dataConnHandle, dataInputBuf, 0, dataInputSize);
        if (rc != MVI_SUCCESS)
        {
            printf("\nMVIcip_WriteConnected failed: %d\n", rc);
            ErrorExit(rc);
        }

        // Wait here until controller acknowledges the new data.
        timeout = 1000;
        while(--timeout)
        {
            rc = MVIcip_ReadConnected(handle, dataConnHandle, (BYTE *)&w, OUT_ACK_OFFSET*2, 2);
            if (rc != MVI_SUCCESS)
            {
                printf("\nMVIcip_ReadConnected failed: %d\n", rc);
                ErrorExit(rc);
            }
            if (w == inputSeq)      // Controller ack?
                break;
            MVIcip_Sleep(handle, 1);
        }
        // If timeout==0, then the processor did not read the
        // message within the timeout period.  Here we just ignore the
        // error.
        
        MVIcip_SetUserLED(handle, MVI_LED_USER1, MVI_LED_STATE_OFF);

        // zero data buffer to get ready for next message
        memset(&portcfg->InBuf, 0, sizeof(portcfg->InBuf));
        portcfg->InBufidx = 0;      // Ready for new serial data
    }
}


/*
** Stack Checking must be disabled for thread routines!!
*/
#ifdef __BORLANDC__
#pragma option -N-
#endif
#ifdef _MSC_VER
#pragma check_stack (off)
#endif

/************************************************************************
*
*     Entry point:                                                      
*       checkTxMsg
*
*     Description:                                                      
*       Check for a message from the PLC to transmit.  If there is
*       a message, send it to the appropriate port, as indicated
*       by the first byte of the message.
*
*     Arguments:                                                        
*       None
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
THREAD __loadds checkTxMsg(void)
{
    WORD len;
    static BYTE tmpBuf[MAX_DATA_SZ+1];
    static WORD outputSeq = 0;
    WORD wPort, newSeq;
    int rc;

    while(RunThread)
    {
        // Read output data sequence to see if new data has arrived
        rc = MVIcip_ReadConnected(handle, dataConnHandle, (BYTE *)&newSeq, OUT_SEQ_OFFSET*2, 2);
        if (rc != MVI_SUCCESS)
            break;                      // if error, exit thread

        if (newSeq != outputSeq)        // New data received from controller?
        {
            // Save the new sequence
            outputSeq = newSeq;

            // Blink the LED for debugging
            MVIcip_SetUserLED(handle, MVI_LED_USER2, MVI_LED_STATE_ON);
        
            // Read the data length
            MVIcip_ReadConnected(handle, dataConnHandle, (BYTE *)&len, OUT_LEN_OFFSET*2, 2);

            // Move the data into the temporary buffer
            MVIcip_ReadConnected(handle, dataConnHandle, tmpBuf, OUT_DATA_OFFSET*2, len);

            // Read the port selection
            MVIcip_ReadConnected(handle, dataConnHandle, (BYTE *)&wPort, OUT_PORT_OFFSET*2, 2);

            // Acknowledge the message - after this, the controller may
            // write new output data at any time
            *(WORD *)&dataInputBuf[INP_ACK_OFFSET*2] = newSeq;
            MVIcip_WriteConnected(handle, dataConnHandle, dataInputBuf, 0, dataInputSize);

            // Send the new data to the appropriate serial port
            switch(wPort)
            {
                case 0:     // PRT1
                    MVIsp_PutData(COM1, tmpBuf, (int *)&len, 1000);
                    break;

                case 1:     // PRT2
                    MVIsp_PutData(COM2, tmpBuf, (int *)&len, 1000);
                    break;

                case 2:     // PRT3
                    MVIsp_PutData(COM3, tmpBuf, (int *)&len, 1000);
                    break;

                default:    // Invalid port number - just ignore
                    break;
            }
            MVIcip_SetUserLED(handle, MVI_LED_USER2, MVI_LED_STATE_OFF);
        }
        PassTimeSlice();    // give the other thread a chance to run
    }

#ifdef DEBUG
    printf("\nThread exiting\n");
#endif

    DeallocateThread();     // thread destructor; never returns.
}

/*
** Return Stack Checking to its default state
*/
#ifdef __BORLANDC__
#pragma option -N.
#endif
#ifdef _MSC_VER
#pragma check_stack ()
#endif

