/**  $Revision:   1.0  $
/**/
/************************************************************************
*
*   Title:  Sample barcode-reader application for the MVI Module
*
*   Abstract:
*
*   This sample program implements a simple interface to a barcode reader
*   or similar device.  It uses the MVI API and transfers data
*   using MVI messaging.
*
*   Environment:    MVI Communications Module
*                   MVI API
*                   General Software DOS 6-XL
*                   Borland/Microsoft C/C++ Compiler (16-bit)
*
*   Author:         JMC
*                   Copyright (c) 1999-2000 Online Development, Inc.
*   
*
*
*   Receiving Serial Data
*   ---------------------
*
*   The first byte of the received message buffer contains the port
*   number from which the data was received (0=PRT1, 1=PRT2, 2=PRT3).
*
*   The termination character is transferred.  Unused bytes at
*   the end of the message buffer are set to zero.
*
*
*   Transmitting Serial Data
*   ------------------------
*
*   Messages received from the processor will be sent to the port
*   specified in the first byte of the message.
*
*
*   Using the program
*   -----------------
*
*   The program allows the baud rate and string termination character
*   to be specified on the command line.  It also allows either or
*   both serial ports to be specified.  For simplicity, the baud and
*   termination character parameters are applied to all serial ports.
*   The ports are setup for 8 bits, no parity, and 1 stop bit.
*
*   Syntax: 
*
*         barcode [-baud=index] [-tc=term char] -p[1|2|3]
*
*   Options:
*
*         -baud=[index]     Specifies serial port baud rate.  Default
*                           is 9600 (index=7).
*
*         -tc=[term char]   Specifies termination character.  ASCII
*                           value of character in decimal is expected.
*                           Default is 10 (line feed).
*
*   At least one of the following must be specified:
*
*         -p1               Indicates that PRT1 is enabled
*
*         -p2               Indicates that PRT2 is enabled
*
*         -p3               Indicates that PRT3 is enabled
*                   
* 
************************************************************************/


/*=======================================================================
=                           INCLUDE FILES                               =
=======================================================================*/

#include    <stdio.h>
#include    <stdlib.h>
#include    <conio.h>
#include    <string.h>
#include    "mvibpapi.h"
#include    "mvispapi.h"


/*=======================================================================
=                    MODULE WIDE DEFINITIONS                            =
=======================================================================*/

#define MAX_DATA_SZ       500   // Maximum string length that can be transferred

// Serial Port Control Structure
typedef struct
{
    int  Enable;                // 1 means port is enabled for barcode data
    int  ComPort;               // Set to COM1, COM2, or COM3
    BYTE TermChar;              // termination character
    WORD StringLength;          // Max string length
    WORD InBufidx;              // Current receive data buffer pointer
    BYTE InBuf[MAX_DATA_SZ+1];  // Message buffer
} PORTCFG;


/*=======================================================================
=                      LOCAL SUPPORT PROTOTYPES                         =
=======================================================================*/

void ErrorExit(int);
void servicePort(PORTCFG *);
void checkTxMsg(void);

/*=======================================================================
=                    MODULE WIDE GLOBAL VARIABLES                       =
=======================================================================*/

MVIHANDLE  handle;          // Handle returned from MVIbp_Open function
MVIBPIOCONFIG IOconfig;     // Module IO configuration
PORTCFG Port1Cfg;           // Allocate data for port 1
PORTCFG Port2Cfg;           // Allocate data for port 2
PORTCFG Port3Cfg;           // Allocate data for port 3


/*=======================================================================
=                       LOCAL SUPPORT ROUTINES                          =
=======================================================================*/

/************************************************************************
*
*     Entry point:                                                      
*       ErrorExit                                           
*
*     Description:                                                      
*       This routine is called whenever an error code is returned from
*       an API routine.  It displays the message associated with the
*       error, then exits the program.
*       
*     Arguments:                                                        
*
*       errcode             : int                           ( input )
*         Error return value from a backplane API function
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void ErrorExit(int errcode)
{
    char errbuf[80];

    if (errcode)
    {
        MVIbp_ErrorString(errcode, errbuf);   // get error message
        printf("%s\n", errbuf);
    }

    MVIbp_Close(handle);                      // should always close before exiting
    exit(1);
}


/************************************************************************
*
*     Entry point:                                                      
*       usage
*
*     Description:                                                      
*       This routine is called when invalid command-line options
*       are encountered.
*       
*     Arguments:                                                        
*
*       name                : char *                        ( input )
*         program name string
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void usage(char *name)
{
    printf("\nUsage: %s [-baud=index] [-tc=term ch] [-setup] -p[1|2|3]", name);
    printf("\nOptions:");
    printf("\n   -baud=index     index specifies baud rate (default=7 or 9600 baud)");
    printf("\n   -tc=term ch     termination character value in decimal (default=10)");
    printf("\n   -setup          do not exit if setup jumper installed");
    printf("\nAt least one of the following must be specified:");
    printf("\n   -p1             enable COM1");
    printf("\n   -p2             enable COM2");
    printf("\n   -p3             enable COM3");
}


/*=======================================================================
=                       MAIN ENTRY POINT                                =
=======================================================================*/

/************************************************************************
*
*     Entry point:                                                      
*       main                                           
*
*     Description:                                                      
*
*     Arguments:                                                        
*       none
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       none
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
*     debugging/error printf's will only be seen if console is enabled.
*
************************************************************************/
void main(int argc, char *argv[])
{
    char *arg;
    int rc;
    int n;
    int mode;
    BYTE Baud = BAUD_9600;      // default baud rate
    BYTE TermChar = _LF;        // default termination character
    int Setup = 0;

    // init port config structs
    memset(&Port1Cfg, 0, sizeof(PORTCFG));
    memset(&Port2Cfg, 0, sizeof(PORTCFG));
    memset(&Port3Cfg, 0, sizeof(PORTCFG));
    Port1Cfg.InBufidx = 1;      // leave first byte for port number
    Port2Cfg.InBufidx = 1;      // leave first byte for port number
    Port3Cfg.InBufidx = 1;      // leave first byte for port number

    // parse the command line options
    for (n=1; n<argc; n++) {    // scan for arguments
        arg = argv[n];

		if (strnicmp("-baud=", arg, 6) == 0) {
            Baud = (BYTE) atoi(arg+6);
            if (Baud > BAUD_115200)
            {
                usage(argv[0]);
                exit(1);
            }
            continue;
        }

		if (strnicmp("-tc=", arg, 4) == 0) {
            TermChar = (BYTE) atoi(arg+4);
            continue;
        }

		if (strnicmp("-setup", arg, 6) == 0) {
            Setup = 1;
            continue;
        }

		if (strnicmp("-p1", arg, 3) == 0) {
            Port1Cfg.Enable = 1;
            continue;
        }

		if (strnicmp("-p2", arg, 3) == 0) {
            Port2Cfg.Enable = 1;
            continue;
        }

		if (strnicmp("-p3", arg, 3) == 0) {
            Port3Cfg.Enable = 1;
            continue;
        }

        usage(argv[0]);
        exit(1);
    }

    // Make sure at least one port is enabled
    if ((!Port1Cfg.Enable) && (!Port2Cfg.Enable) && (!Port3Cfg.Enable))
    {
        usage(argv[0]);
        exit(1);
    }

    // Open the Backplane API
    if (MVI_SUCCESS != (rc = MVIbp_Open(&handle)))
    {
        printf("\nMVIbp_Open failed: %d\n", rc);
        ErrorExit(rc);
    }

    // Check for setup mode - if so, exit now unless setup option is present.
    // The purpose for this is to allow the user to regain control of the
    // module by installing the Setup Jumper.  In a typical configuration,
    // the module application is run from AUTOEXEC.BAT.  In this case,
    // we exit the application if the Setup Jumper is installed, unless
    // the -setup command line option is preset.  This option is useful
    // for application debugging.
    MVIbp_GetSetupMode(handle, &mode);
    if (!Setup && mode)
    {   // Print the banner and version only if setup jumper is on
        printf("\nMVI Barcode Sample Application (MVI API example)");
        printf("\nCopyright (c) 2000 Online Development, Inc.\n");
        printf("\nSetup jumper installed: exiting application\n");
        MVIbp_Close(handle);
        exit(0);
    }

    // Query the module IO configuration (module type setup in BIOS)
    if (MVI_SUCCESS != (rc = MVIbp_GetIOConfig(handle, &IOconfig)))
    {
        printf("\nMVIbp_GetIOConfig failed: %d\n", rc);
        ErrorExit(rc);
    }

    // Initialize the serial port(s)
    if (Port1Cfg.Enable)
    {
        rc = MVIsp_Open(COM1, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port1Cfg.ComPort = COM1;
        Port1Cfg.StringLength = (IOconfig.MsgRcvBufSize * 2) - 1;
        Port1Cfg.TermChar = TermChar;
    }
    if (Port2Cfg.Enable)
    {
        rc = MVIsp_Open(COM2, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port2Cfg.ComPort = COM2;
        Port2Cfg.StringLength = (IOconfig.MsgRcvBufSize * 2) - 1;
        Port2Cfg.TermChar = TermChar;
    }
    if (Port3Cfg.Enable)
    {
        rc = MVIsp_Open(COM3, Baud, PARITY_NONE, WORDLEN8, STOPBITS1);
        if (rc != MVI_SUCCESS)
            ErrorExit(rc);
        Port3Cfg.ComPort = COM3;
        Port3Cfg.StringLength = (IOconfig.MsgRcvBufSize * 2) - 1;
        Port3Cfg.TermChar = TermChar;
    }

    // Main Loop - never exits
    while(1)
    {
        servicePort(&Port1Cfg);     // service rx data
        servicePort(&Port2Cfg);     // service rx data
        servicePort(&Port3Cfg);     // service rx data
        checkTxMsg();               // Look for message to transmit
    }

// The following line is needed if the loop above is allowed to exit.
//    MVIbp_Close(handle);

}


/************************************************************************
*
*     Entry point:                                                      
*       servicePort                                           
*
*     Description:                                                      
*       Check for data from the serial port.  If data is ready, move
*       it to the port's buffer.  If the message is complete (either
*       by matching the termination character or the string length),
*       then send the data to the PLC.
*
*     Arguments:                                                        
*
*       portcfg             : PORTCFG *                     ( input )
*         Pointer to port control structure
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void servicePort(PORTCFG *portcfg)
{
    int rc;
    int len;

    // Return if port is not enabled 
    if (!portcfg->Enable)
        return;

    // Calculate number of characters left to go
    len = portcfg->StringLength - portcfg->InBufidx;

    rc = MVIsp_Gets(portcfg->ComPort, &portcfg->InBuf[portcfg->InBufidx],
             portcfg->TermChar, &len, TIMEOUT_ASAP);

    if (len == 0)                       // Return if no data in queue
        return;

    // Update buffer index
    portcfg->InBufidx += len;

    // Termination test - see if string is complete
    if (rc == MVI_SUCCESS)
    {
        // String is complete - build a message and send to PLC 

        // Write port number to buffer
        portcfg->InBuf[0] = portcfg->ComPort;

        // Write data for controller to fetch - blink LED for debugging
        MVIbp_SetUserLED(handle, MVI_LED_USER1, MVI_LED_STATE_ON);
        rc = MVIbp_SendMessage(handle, (WORD *)portcfg->InBuf, 
                (portcfg->InBufidx/2)+1, 0, 2000);
        // If rc == MVI_ERR_TIMEOUT, then the processor did not read the
        // message within the timeout period.  Here we just ignore the
        // error.
        
        MVIbp_SetUserLED(handle, MVI_LED_USER1, MVI_LED_STATE_OFF);

        portcfg->InBufidx = 1;      // Ready for new serial data
        // zero data buffer to get ready for next message
        memset(&portcfg->InBuf, 0, sizeof(portcfg->InBuf));
    }
}


/************************************************************************
*
*     Entry point:                                                      
*       checkTxMsg
*
*     Description:                                                      
*       Check for a message from the PLC to transmit.  If there is
*       a message, send it to the appropriate port, as indicated
*       by the first byte of the message.
*
*     Arguments:                                                        
*       None
*
*     External effects:                                                 
*
*
*     Return value:                                                     
*       None
*
*-----------------------------------------------------------------------
*     Notes:                                                            
*
************************************************************************/
void checkTxMsg(void)
{
    int rc;
    WORD len = MAX_DATA_SZ;
    static BYTE tmpBuf[MAX_DATA_SZ+1];

    rc = MVIbp_ReceiveMessage(handle, (WORD *)tmpBuf, &len, 0, 0);
    if (rc == MVI_SUCCESS)          // did we get a message?
    {
        // Blink the LED for debugging
        MVIbp_SetUserLED(handle, MVI_LED_USER2, MVI_LED_STATE_ON);
        len = len*2-1;
        switch(tmpBuf[0])
        {
            case 0:     // PRT1
                MVIsp_PutData(COM1, &tmpBuf[1], (int *)&len, 1000);
                break;

            case 1:     // PRT2
                MVIsp_PutData(COM2, &tmpBuf[1], (int *)&len, 1000);
                break;

            case 2:     // PRT3
                MVIsp_PutData(COM3, &tmpBuf[1], (int *)&len, 1000);
                break;

            default:    // Invalid port number - just ignore
                break;
        }
        MVIbp_SetUserLED(handle, MVI_LED_USER2, MVI_LED_STATE_OFF);
    }
}


