	TITLE	INTLAT - Interrupt latency test program.

;***	INTLAT - Interrupt latency test program.
;
;1.	Functional Description.
;	This program hooks IRQ 0, the highest priority interrupt on the PC.
;	When hooked, it takes samples of interrupt latency until a key is
;	pressed, at which time the minimum, maximum, and average interrupt
;	latency is displayed.  This latency is calculated by sampling the
;	8254's timer registers directly immediately after an interrupt occurs.
;	Because the timer generates an interrupt when it counts down to 0,
;	it is possible to measure how long it took from timer countdown to
;	the actual interrupt being serviced.
;
;2.	Modification History.
;	S. E. Jones	93/02/20.	Original.
;
;3.	NOTICE: Copyright (C) 1993 General Software.
;
;4.	Build Environment.
;	MASM 5.10, no special switches.

	include ..\inc\usegs.inc
	include ..\inc\udefines.inc
	include ..\inc\umacros.inc
	include ..\inc\ustruc.inc

;	Define the stack.

_STACK	SEGMENT PARA STACK 'STACK'
	db	512 dup ('$')
TopStack =	$
_STACK	ENDS

UCODE	SEGMENT

OldIsr08   dd	?
MinLatency dw	65535
MaxLatency dw	0
AvgLatency dw	0			; using the successive averaging method.
Counts	   dd	0			; number of interrupts.

;***	Delay - Delay for Back-to-Back I/O.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine just delays for back-to-back I/O to work.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	91/12/16.	Original.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	none.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc Delay
EndProc Delay

;***	ReadTime - Return time in ticks.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine returns the BIOS timestamp in 100us units.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	91/12/16.	Original.
;	S. E. Jones	92/07/03.	Added Delay calls.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	flags.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc ReadTime
	push	ax

;	Read the 8253's channel 0 timer value, which is in 0.8380966us units.

	cli				; don't allow latency here.
	mov	al, 00000000b		; (AL) = latch command.
	out	43h, al                 ; latch the data.
	Pcall	Delay			; allow I/O port to recover.
	in	al, 40h                 ; now read the ports.
	Pcall	Delay			; allow I/O port to recover.
	mov	ah, al
	in	al, 40h
	xchg	ah, al			; (AX) = 16-bit timer 0 data.
	neg	ax			; count-up, not down.

;	Accumulate totals.

	add	AvgLatency, ax		; add-in our value.
	shr	AvgLatency, 1		; divide sum by 2 to get average.

	cmp	ax, MinLatency		; is this smaller than our best min?
	jae	@f			; if not.
	mov	MinLatency, ax		; if so, replace with new best.
@@:

	cmp	ax, MaxLatency		; is this larger than our best max?
	jbe	@f			; if not.
	mov	MaxLatency, ax		; if so, replace with new best.
@@:
	add	Counts.lo, 1		; count number of samples.
	adc	Counts.hi, 0		; use 32-bit arithmetic.
	pop	ax
EndProc ReadTime

;***	NewIsr8 - Clock Entrypoint.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is executed as an ISR for IRQ 0, the system clock tick.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	92/02/19.	Original.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	none.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc NewIsr8, PUBLIC, FAR
	Pcall	ReadTime		; sample interrupt latency.
	jmp	dword ptr cs:[OldIsr08] ; pass control to the BIOS.
EndProc NewIsr8

;***	Main - Main Entrypoint.
;
;   FUNCTIONAL DESCRIPTION.
;	This routine is the entrypoint of the test program.  We print
;	a sign-on banner, and begin the loop we're testing.
;
;   MODIFICATION HISTORY.
;	S. E. Jones	92/02/19.	Original.
;
;   WARNINGS.
;	none.
;
;   ENTRY.
;	none.
;
;   EXIT.
;	none.
;
;   USES.
;	all.

	ASSUME	CS:CGROUP, DS:NOTHING, ES:NOTHING, SS:NOTHING
DefProc Main, PUBLIC, FAR
	mov	ax, DGROUP
	mov	ds, ax
	ASSUME	DS:DGROUP		; (DS) = DGROUP.
	PRINTF	<Interrupt latency test running, press any key to stop.\n>

;	Hook IRQ 0's vector so we can get control.

	sub	ax, ax
	mov	es, ax
	ASSUME	ES:NOTHING
	mov	ax, es:[08h*4+0]	; (AX) = offset portion of vector.
	mov	word ptr cs:[OldIsr08+0], ax ; save it.
	mov	ax, es:[08h*4+2]	; (AX) = segment portion of vector.
	mov	word ptr cs:[OldIsr08+2], ax ; save that, too.

	cli				; BEGIN HARD CRITICAL SECTION.
	mov	word ptr es:[08h*4+0], OFFSET CGROUP:NewIsr8
	mov	word ptr es:[08h*4+2], cs
	sti				; END HARD CRITICAL SECTION.

;	Now enter the main loop, waiting for a character to be pressed.

Main_Loop:
	mov	ah, 1
	int	16h
	jz	Main_Loop		; if char not available.

	mov	ah, 0
	int	16h			; eat char.

;	Restore old interrupt vector.

	sub	ax, ax
	mov	es, ax
	ASSUME	ES:NOTHING
	cli				; BEGIN HARD CRITICAL SECTION.
	mov	ax, word ptr cs:[OldIsr08+0] ; (AX) = saved offset of old vector.
	mov	es:[08h*4+0], ax	; restore it.
	mov	ax, word ptr cs:[OldIsr08+2] ; (AX) = saved segment portion of vector.
	mov	es:[08h*4+2], ax	; restore it, too.
	sti				; END HARD CRITICAL SECTION.

;	Now display our findings.

	PRINTF	<Total samples taken: $lu.\n>, <Counts.hi, Counts.lo>
	PRINTF	<Minimum latency:     $u * 0.838us.\n>, <MinLatency>
	PRINTF	<Maximum latency:     $u * 0.838us.\n>, <MaxLatency>
	PRINTF	<Average latency:     $u * 0.838us.\n>, <AvgLatency>

;	Now terminate the process and return to DOS.

	mov	ah, 4ch
	mov	al, 0			; successful status code.
	int	21h			; terminate program.
EndProc Main

UCODE	ENDS
	END	Main
