//***	IORP.H -- I/O Request Packet Structures for C Device Drivers.
//
//1.	Functional Description.
//	This include file contains structure definitions for use by
//	Embedded DOS device drivers written in C.
//
//2.	Modification History.
//	S. E. Jones	95/03/17.	#6-XL, original.
//
//3.	NOTICE: Copyright (C) 1995 General Software, Inc.

#ifndef _IORPDEF_
#define _IORPDEF_

#include "../inc/bpb.h"

//
// IORP command codes.
//

#define DEV_INITIALIZE		0	// initialize driver.
#define DEV_MEDIA_CHECK         1	// media check on block device.
#define DEV_BUILD_BPB		2	// build BIOS Parameter Block.
#define DEV_IOCTL_READ		3	// I/O Control read.
#define DEV_READ		4	// read from device.
#define DEV_ND_READ		5	// non-destructive read.
#define DEV_INPUT_STATUS	6	// return current input status.
#define DEV_INPUT_FLUSH         7	// flush device input buffer.
#define DEV_WRITE		8	// write to device.
#define DEV_WRITE_VFY		9	// write with verify.
#define DEV_OUTPUT_STATUS	10	// return current output status.
#define DEV_OUTPUT_FLUSH	11	// flush device output buffer.
#define DEV_IOCTL_WRITE         12	// I/O Control write.
#define DEV_OPEN		13	// device open.
#define DEV_CLOSE		14	// device close.
#define DEV_REMOVE_MEDIA	15	// removable media.
#define DEV_OUTPUT_BUSY         16	// output until busy.
#define DEV_GENERIC_IOCTL	19	// generic IOCTL function.
#define DEV_GET_LOGICAL_DEVICE	23	// get logical device mapping.
#define DEV_SET_LOGICAL_DEVICE	24	// set logical device mapping.
#define DEV_LIMIT (DEV_SET_LOGICAL_DEVICE) // last possible command code.

//
// IORP_HEADER - I/O Request Packet Header.
//
// This structure defines the layout of an I/O Request Packet header,
// used as the basic control block to pass to device drivers.
//

typedef struct _IORP {
    UCHAR Length;			// length of request header.
    UCHAR Unit;                         // unit number of this request.
    UCHAR Command;			// command code.
    USHORT Status;			// device driver's return status code.
    UCHAR Reserved [8];                 // reserved for EDOS I/O system.
} IORP, *PIORP;

//
// IORP status bitflags and field masks.
//

#define IORPSTATUS_ERROR	0x8000	// indicates error.
#define IORPSTATUS_RETMASK	0x00ff	// error code if IORPSTATUS_ERROR.
#define IORPSTATUS_DONE         0x0100	// if operation complete.
#define IORPSTATUS_BUSY         0x0200	// if operation in progress.

#define DEVERR_WRITE_PROTECT_VIO 0	// write protection violation.
#define DEVERR_UNKNOWN_UNIT	 1	// unknown unit number.
#define DEVERR_DRIVE_NOT_READY	 2	// specified drive not closed/ready.
#define DEVERR_BAD_COMMAND	 3	// bad command, driver cannot process.
#define DEVERR_DATA_ERROR	 4	// CRC failure; data corruption.
#define DEVERR_BAD_STRUC_LENGTH  5	// bad drive request structure length.
#define DEVERR_SEEK_FAILURE	 6	// seek failure on drive.
#define DEVERR_UNKNOWN_MEDIA	 7	// media unknown.
#define DEVERR_SECTOR_NOT_FOUND  8	// sector number could not be found.
#define DEVERR_PAPER_OUT	 9	// printer out of paper.
#define DEVERR_WRITE_FAULT	 10	// write fault.
#define DEVERR_READ_FAULT	 11	// read fault.
#define DEVERR_GENERAL_FAILURE	 12	// catch-all category for hard errors.
#define DEVERR_INVALID_DISK_CHANGE 15	// (only on DOS versions 3 and above).

//
// This structure defines the layout of an IORP for the Init Function.
//

typedef struct _IORP_INIT {
    IORP RequestHeader;                 // basic request structure.
    UCHAR NumberOfUnits;		// unit count/this driver.
    VOID *BreakAddress;                 // ptr to 1st byte following driver.
    VOID *BpbPointer;			// ptr to BPB array/this driver.
    USHORT FirstUnit;			// 1st unit assigned to driver.
} IORP_INIT, *PIORP_INIT;

#define IORP_INIT_CMDTAIL  BpbPointer	// IN ptr to DEVICE= string.

//
// IORP_MEDIA_CHECK - I/O Request Packet for Media Check Function.
//

typedef struct _IORP_MEDIA_CHECK {
    IORP RequestHeader;                 // basic request structure.
    UCHAR MediaDescriptor;		// 0xf0, 0xf8, etc.
    UCHAR ChangedFlag;			// OUT status, as follows.

#define IORP_MC_CHANGEFLAG_CHANGED -1	// if it has changed for sure.
#define IORP_MC_CHANGEFLAG_UNKNOWN 0	// if we don't really know.
#define IORP_MC_CHANGEFLAG_SAFE    1	// if it definitely hasn't changed.

    UCHAR *VolumeId;			// ptr to prev volume ID. (DOS 3.0).
} IORP_MEDIA_CHECK, *PIORP_MEDIA_CHECK;

//
// IORP_BUILD_BPB - I/O Request Packet for Build BPB Function.
//

typedef struct _IORP_BUILD_BPB {
    IORP RequestHeader;                 // basic request structure.
    UCHAR MediaDescriptor;		// 0xf0, 0xf8, etc.
    VOID *TransferAddress;		// ptr to transfer area.
    PBPB NewBpb;			// OUT ptr to new BIOS parameter block.
} IORP_BUILD_BPB, *PIORP_BUILD_BPB;

//
// IORP_IO - I/O Request Packet for Read and Write Functions.
//

typedef struct _IORP_IO {
    IORP RequestHeader;                 // basic request structure.
    UCHAR MediaDescriptor;		// 0xf0, 0xf8, etc.
    VOID *TransferAddress;		// ptr to transfer area.
    USHORT NumberOfSectors;		// number of sectors to xfer.
    USHORT SectorNumber16;		// 16-bit sector number.

#define USE_32BIT_SECTOR_NUMBER 0xffff	// set to this to use SectorNumber32.

    //
    // The following fields are extensions of the IORP if the block driver
    // supports LSN's bigger than 65535.  It tells DOS this by setting the
    // DEVATR_BIGDISK bit in its device attributes word.  These fields are
    // ignored by other drivers.

    ULONG VolumeId;			// OUT volume identifier.
    ULONG SectorNumber32;		// 32-bit LSN if Sectno==0xffff.
} IORP_IO, *PIORP_IO;

//
// IORP_IOCTL - I/O Request Packet for IOCTL Function.
//
// This structure defines the layout of an IORP for the generic IOCTL.
// function code. This IS NOT the same as IOCTL Read or IOCTL Write.
//

typedef struct _IORP_IOCTL {
    IORP RequestHeader;                 // basic request structure.
    UCHAR CategoryCode;                 // major (category) code.
    UCHAR FunctionCode;                 // minor (subfunction) code.
    USHORT SiContents;			// contents of user's (SI) register.
    USHORT DiContents;			// contents of user's (DI) register.
    VOID *ParameterBlock;		// ptr, parameter block.
} IORP_IOCTL, *PIORP_IOCTL;

#define GENERIC_SET_DEVICE_PARAMS  0x40
#define GENERIC_WRITE_TRACK	   0x41
#define GENERIC_FORMAT_AND_VERIFY  0x42
#define GENERIC_GET_DEVICE_PARAMS  0x60
#define GENERIC_READ_TRACK	   0x61
#define GENERIC_VERIFY		   0x62

//
// DEVHDR Installable Device Driver Header.
//

typedef struct _DEVHDR {
    VOID *NextHeader;			// ptr, next header in system.

#define NULL_HEADER	0xffffffffL	// ptr to no next header.

    USHORT Flags;			// attribute flags, below.

#define DEVATR_CHAR		0x8000	// device is a character device.
#define DEVATR_IOCTL		0x4000	// device can handle IOCTL read/write calls.
#define DEVATR_NONIBMBLK	0x2000	// block device, but non-IBM format.
#define DEVATR_OUTBUSY		0x2000	// OVERLOADS!, output_until_busy supported.
#define DEVATR_EXTENDED         0x1000	// if set, this is an EDEVHDR.
#define DEVATR_OPNCLS		0x0800	// open/close supported (3.x only).
#define DEVATR_V32		0x0040	// generic IOCTL, get/set log drive supported.
#define DEVATR_INT29		0x0010	// supports fast CON handler (int 29h).
#define DEVATR_CLOCK		0x0008	// device is CLOCK.
#define DEVATR_NUL		0x0004	// device is NUL.
#define DEVATR_BIGDISK		0x0002	// block device uses 32-bit sector numbers.
#define DEVATR_CONOUT		0x0002	// device is console output.
#define DEVATR_CONIN		0x0001	// device is console input.

    USHORT StrategyEpt;                 // code offset, strategy entrypoint.
    USHORT InterruptEpt;		// code offset, interrupt entrypoint.

    UCHAR NumberOfUnits;		// number of units supported by driver.

} DEVHDR, *PDEVHDR;

#endif // _IORPDEF_
